/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/commands/ChangeIdCommand
 */
define(['baja!',
        'lex!bacnet',
        'jquery',
        'Promise',
        'underscore',
        'dialogs',
        'bajaux/commands/Command',
        'nmodule/bacnet/rc/wb/mgr/model/columns/DeviceIdMgrColumn',
        'nmodule/webEditors/rc/fe/feDialogs',
        'nmodule/webEditors/rc/wb/mgr/BatchComponentEditor',
        'nmodule/webEditors/rc/wb/mgr/mgrUtils',
        'nmodule/webEditors/rc/wb/mgr/model/MgrModel',
        'nmodule/webEditors/rc/wb/mgr/model/columns/NameMgrColumn',
        'nmodule/webEditors/rc/wb/table/model/Column',
        'nmodule/webEditors/rc/wb/table/model/source/ArrayComponentSource',
        'baja!bacnet:ChangeDeviceIdConfig'], function (
        baja,
        lexs,
        $,
        Promise,
        _,
        dialogs,
        Command,
        DeviceIdMgrColumn,
        feDialogs,
        BatchComponentEditor,
        mgrUtils,
        MgrModel,
        NameMgrColumn,
        Column,
        ArrayComponentSource) {

  'use strict';

  var bacnetLex = lexs[0],
      getTableSelection = mgrUtils.getMainTableSelectedSubjects;

////////////////////////////////////////////////////////////////
// ChangeIdBatchComponentEditor
////////////////////////////////////////////////////////////////

  /**
   * ChangeIdBatchComponentEditor is a customized version of the batch component editor
   * that will not commit the modified device ids back to the components in the model,
   * but will instead set them as properties on a configuration object that will be
   * passed as the parameter to a job.
   *
   * @private
   * @constructor
   */
  var ChangeIdBatchComponentEditor = function ChangeIdBatchComponentEditor() {
    BatchComponentEditor.apply(this, arguments);
  };

  ChangeIdBatchComponentEditor.prototype = Object.create(BatchComponentEditor.prototype);
  ChangeIdBatchComponentEditor.prototype.constructor = ChangeIdBatchComponentEditor;

  /**
   * Override of doInitialize() sets an extra CSS class, that allows us to configure
   * the UI slightly differently.
   *
   * @param {JQuery} dom
   * @returns {*}
   */
  ChangeIdBatchComponentEditor.prototype.doInitialize = function (dom) {
    dom.addClass('ChangeIdBatchComponentEditor');
    return BatchComponentEditor.prototype.doInitialize.apply(this, arguments);
  };

  /**
   * Overridden version of the doSave() function; we don't want the changed values
   * committed back to the source components, we want to obtain the values to set
   * as old/new properties on a configuration component.
   *
   * @returns {Promise}
   */
  ChangeIdBatchComponentEditor.prototype.doSave = function () {
    return Promise.resolve();
  };

  /**
   * Create a manager model for showing in the component editor. This contains
   * columns for the device name and device id.
   *
   * @static
   * @param {Array.<baja.Component>} comps - the selected device components
   * @returns {module:nmodule/webEditors/rc/wb/mgr/model/MgrModel}
   */
  ChangeIdBatchComponentEditor.makeEditModel = function (comps) {
    var columns = [
      new NameMgrColumn({ flags: Column.flags.READONLY | Column.flags.EDITABLE }),
      new DeviceIdMgrColumn('config/deviceObject/objectId', {
        displayName: bacnetLex.get('deviceManager.deviceId'),
        type: baja.lt('bacnet:BacnetDeviceObject'),
        flags: Column.flags.EDITABLE
      })
    ];

    return new MgrModel({
      componentSource: new ArrayComponentSource(comps),
      columns: columns
    });
  };

////////////////////////////////////////////////////////////////
// ChangeIdCommand
////////////////////////////////////////////////////////////////

  function appendValueString(str, currentId, newId) {
    if (str.length) { str = str + ';'; }
    return str + currentId + '->' + newId;
  }

  /**
   * Called to save the proposed values from the ChangeIdBatchComponentEditor
   * on a new instance of a ChangeDeviceIdConfig component. This is the component
   * that will be submitted as the argument when creating the job to change the
   * ids.
   *
   * @returns {baja.Component} a ChangeDeviceIdConfig instance.
   */
  function createConfig(model) {
    var cfg = baja.$('bacnet:ChangeDeviceIdConfig'),
        devIdColumn = model.getColumn('objectId'),
        str = '';

    _.map(model.getRows(), function (row) {
      var currentId = devIdColumn.getValueFor(row),
          newId = devIdColumn.getProposedValueFor(row);

      str = appendValueString(str, currentId, newId);

      cfg.add({
        slot: 'currentId?',
        value: currentId,
        flags: baja.Flags.READONLY
      });

      cfg.add({
        slot: 'changeTo?',
        value: newId,
        flags: baja.Flags.READONLY
      });
    });

    cfg.setValue(str);

    return cfg;
  }

  /**
   * `Command` allowing the user to change the device id instance number
   * of one or more devices in the manager table.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/wb/mgr/commands/ChangeIdCommand
   * @extends module:bajaux/commands/Command
   */
  var ChangeIdCommand = function ChangeIdCommand(mgr) {
    var that = this;

    Command.call(this, {
      displayName: bacnetLex.get('changeId.label'),
      description: bacnetLex.get('changeId.description'),
      icon: bacnetLex.get('changeId.icon'),
      enabled: false,

      /**
       * For the devices currently selected in the main table, display a
       * batch component editor, with columns for the device name and device
       * id. The values from the id column are used to create slots on a
       * ChangeDeviceIdConfig instance.
       *
       * @alias module:nmodule/bacnet/rc/wb/mgr/commands/ChangeIdCommand#invoke
       * @returns {Promise}
       */
      func: function () {
        var network = mgr.getNetwork();

        dialogs.showYesNo({
          title: bacnetLex.get('changeId.title'),
          content: bacnetLex.getSafe('changeId.confirm'),
          yes: function () {
            var selection = getTableSelection(mgr),
                model;

            if (selection.length && selection[0].getType().is('bacnet:BacnetDevice')) {
              model = ChangeIdBatchComponentEditor.makeEditModel(selection);

              mgr.getOrdBase()
                .then(function (ordBase) {
                  return feDialogs.showFor({
                    value: model,
                    properties: { ordBase: ordBase },
                    type: ChangeIdBatchComponentEditor
                  });
                })
                .then(function (editModel) {
                  if (editModel) {
                    var config = createConfig(editModel);
                    return network.submitDeviceManagerJob(config)
                      .then(function (ord) {
                        mgr.setJob(baja.Ord.make({
                          base: baja.Ord.make('station:'),
                          child: ord.relativizeToSession()
                        }));
                      });
                  }
                })
                .catch(baja.error);
            }
          }
        });
      }
    });

    mgr.on('jobcomplete', function (job) {
      if (job.getType().is('bacnet:ChangeDeviceIdJob')) {
        that.complete(job);
      }
    });
  };

  /**
   * Function to obtain the text to be displayed once the change id job has completed,
   * either some generic success text or a failure message based on a string obtained
   * from the job.
   *
   * @param {baja.Component} job - the ChangeDeviceIdJob created by invoking this command
   * @returns {String} the string to be displayed in the dialog.
   */
  function getCompletionMessage(job) {
    var cause;
    if (job.getJobState().getTag() === 'success') {
      return bacnetLex.getSafe('changeId.success');
    } else {
      cause = job.get('failureCause') || '';
      return bacnetLex.getSafe({ key: 'changeId.fail', args: [ cause ] });
    }
  }

  ChangeIdCommand.prototype = Object.create(Command.prototype);
  ChangeIdCommand.prototype.constructor = ChangeIdCommand;

  /**
   * Function called when the submitted job completes. Will display a message to the
   * user, informing them of success or failure.
   *
   * @static
   * @param {baja.Component} job
   */
  ChangeIdCommand.prototype.complete = function (job) {
    dialogs.showOk({
      title: bacnetLex.get('changeId.title'),
      content: getCompletionMessage(job)
    });
  };

  return (ChangeIdCommand);
});
