/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/wb/mgr/BacnetConfigUxManager
 */
define([
  'baja!',
  'baja!' +
    'bacnet:BacnetAnalogInput,' +
    'bacnet:BacnetConfigDeviceExt,' +
    'bacnet:BacnetConfigFolder,' +
    'bacnet:DiscoveryConfig,' +
    'bacnet:BacnetObject',
  'Promise',
  'underscore',
  'nmodule/bacnet/rc/wb/mgr/model/BacnetConfigLearnModel',
  'nmodule/bacnet/rc/wb/mgr/model/BacnetConfigMgrModel',
  'nmodule/driver/rc/wb/mgr/DriverMgr',
  'nmodule/webEditors/rc/fe/baja/util/compUtils',
  'nmodule/webEditors/rc/wb/mgr/MgrLearn',
  'nmodule/webEditors/rc/wb/mgr/MgrTypeInfo',
  'nmodule/webEditors/rc/wb/tree/TreeNode'
], function (
  baja,
  types,
  Promise,
  _,
  BacnetConfigLearnModel,
  BacnetConfigMgrModel,
  DriverMgr,
  compUtils,
  addMgrLearnSupport,
  MgrTypeInfo,
  TreeNode) {

  'use strict';

  var closest = compUtils.closest,

      DEVICE_EXT_TYPE_SPEC = 'bacnet:BacnetConfigDeviceExt',
      FOLDER_TYPE_SPEC = 'bacnet:BacnetConfigFolder',

      LEARN_TABLE_NODE_ICON = baja.Icon.make([
        'module://bacnet/com/tridium/bacnet/ui/icons/bacObject.png'
      ]);

  //////////////////////////////////////////////////////////////////////////////
  // BacnetConfigUxManager
  //////////////////////////////////////////////////////////////////////////////

  /**
   * BacnetConfigUxManager constructor.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/wb/mgr/BacnetConfigUxManager
   * @extends module:nmodule/driver/rc/wb/mgr/DriverMgr
   * @param {Object} params
   */
  var BacnetConfigUxManager = function BacnetConfigUxManager(params) {
    DriverMgr.call(this, _.extend({
      moduleName: 'bacnet',
      keyName: 'BacnetConfigUxManager',
      folderType: FOLDER_TYPE_SPEC,
      editableTypes: [ baja.lt('bacnet:BacnetObject') ],
      subscriptionDepth: 2
    }, params));

    addMgrLearnSupport(this);
  };
  BacnetConfigUxManager.prototype = Object.create(DriverMgr.prototype);
  BacnetConfigUxManager.prototype.constructor = BacnetConfigUxManager;

  /**
   * Overrides the driver manager's doInitialize() function.
   *
   * @param {JQuery} dom
   * @returns {*}
   */
  BacnetConfigUxManager.prototype.doInitialize = function (dom) {
    var that = this;

    that.on('jobcomplete', function (job) {
      if (job.getType().is('bacnet:BacnetDiscoverConfigJob')) {
        that.$discoveryJobComplete(job, that).catch(baja.error);
      }
    });

    return DriverMgr.prototype.doInitialize.apply(that, arguments);
  };
  /**
   * Makes the model for the main database table.
   *
   * @param {baja.Component} component - a `bacnet:BacnetConfigDeviceExt` or
   * `bacnet:BacnetConfigFolder` instance.
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/model/MgrModel>}
   */
  BacnetConfigUxManager.prototype.makeModel = function (component) {
    return BacnetConfigMgrModel.make(component);
  };

  /**
   * Make the model for the discovery table.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>}
   */
  BacnetConfigUxManager.prototype.makeLearnModel = function () {
    return BacnetConfigLearnModel.make();
  };

    /**
   * Returns the config device ext for the current loaded component. As we may
   * have a folder loaded, this may be several levels up in the component tree.
   *
   * @returns {baja.Component}
   */
  BacnetConfigUxManager.prototype.getDeviceExt = function () {
    return closest(this.value(), DEVICE_EXT_TYPE_SPEC);
  };

  ////////////////////////////////////////////////////////////////
  // Discovery
  ////////////////////////////////////////////////////////////////

  /**
   * Make a tree node for an item in the discovery table. Device discovery
   * nodes never have any children.
   *
   * @param  {Object} discovery - a discovered object.
   * @returns {module:nmodule/webEditors/rc/wb/tree/TreeNode}
   */
  function makeDiscoveryTableNode(discovery) {
    var name = discovery.getObjectName(),
        node = new TreeNode(name, name);

    node.getIcon = _.constant(LEARN_TABLE_NODE_ICON);
    node.mayHaveKids = _.constant(false);
    node.value = _.constant(discovery);

    return node;
  }

  /**
   * Reload the learn table from the manager's discovered configs.
   *
   * @param {module:nmodule/bacnet/rc/wb/mgr/BacnetConfigUxManager} mgr - the
   * BACnet config manager
   * @returns {Promise}
   */
  function reloadLearnTable(mgr) {
    var model = mgr.getLearnModel(),
        currentCount = model.getRows().length,
        root = model.getRootNode();

    return Promise.map(mgr.$discoveries || [], function (discovery) {
      return makeDiscoveryTableNode(discovery);
    })
      .then(function (nodes) {
        // TODO: create an 'updateLearnRoots' function in the discovery mixin to
        // make this functionality available elsewhere.
        return Promise.map(root.getKids(), function (kid) {
          return root.remove(kid);
        })
          .then(function () {
            return Promise.map(nodes, function (node) {
              return root.add(node);
            });
          })
          .then(function () {
            var promise = currentCount ?
                  (model.removeRows(0, currentCount)) :
                  Promise.resolve();
            return promise.then(function () {
              return model.insertRows(nodes, 0);
            });
          });
      });
  }

  /**
   * Called when the discovery job completes, this will get the discovered
   * items from the job and reload the table.
   *
   * @private
   * @param {baja.Component} job - the config discovery job.
   */
  BacnetConfigUxManager.prototype.$discoveryJobComplete = function (job) {
    var that = this;

    return job.loadSlots()
      .then(function () {
        var props = job.getSlots().properties();
        that.$discoveries = props.is('bacnet:DiscoveryConfig').toValueArray();

        return reloadLearnTable(that);
      });
  };

  /**
   * Start the discovery job.
   *
   * @returns {Promise}
   */
  BacnetConfigUxManager.prototype.doDiscover = function () {
    var that = this;

    return that.getDeviceExt().submitConfigDiscoveryJob()
      .then(function (ord) {
        ord = baja.Ord.make({
          base: baja.Ord.make('station:'),
          child: ord.relativizeToSession()
        });

        return that.setJob(ord);
      });
  };

  /**
   * Return the types available to be created from a given discovery item.
   *
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>}
   */
  BacnetConfigUxManager.prototype.getTypesForDiscoverySubject = function (discovery) {
    return MgrTypeInfo.make(discovery.getTypeSpecs());
  };

  /**
   * Get the values to be set as the proposals on the batch component editor for
   * the rows being added via the `AddCommand`.
   *
   * @param {baja.Complex} discovery - a DiscoveryConfig struct instance.
   * @returns {Object} - an object populated with properties to be used for the
   * new row.
   */
  BacnetConfigUxManager.prototype.getProposedValuesFromDiscovery = function (discovery) {
    return {
      name: discovery.get('objectName'),
      values: {
        objectId: discovery.get('objectId'),
        value: discovery.get('value'),
        description: discovery.get('description')
      }
    };
  };

  /**
   * Test whether the given object matches the component in the station.
   * The comparison is based on the bacnet object identifier.
   *
   * @param discovery
   *
   * @param {baja.Component} comp
   *
   * @returns {boolean} - true if the discovery item and station component
   * match.
   */
  BacnetConfigUxManager.prototype.isExisting = function (discovery, comp) {
    return discovery.get('objectId').equals(comp && comp.get('objectId'));
  };

  ////////////////////////////////////////////////////////////////
  // State
  ////////////////////////////////////////////////////////////////

  /**
   * Save the state we require for restoring after a hyperlink or page reload.
   * In addition to the visible columns saved by the base type, this will save
   * the last set of discovered devices and the ord of any job we may be displaying
   * in the job bar.
   *
   * @returns {Object} an object with the state to be persisted
   */
  BacnetConfigUxManager.prototype.saveStateForOrd = function () {
    var that = this, obj = {};

    if (that.$discoveries) {
      obj.$discoveries = _.map(that.$discoveries, function (discovery) {
        return baja.bson.encodeValue(discovery);
      });
    }

    return obj;
  };

  /**
   * Restore the state we saved in the call to saveStateForOrd(). This
   * will reinstate the learn model from any BDiscoveryDevice instances
   * we persisted.
   *
   * @param {Object} obj - an object with data retrieved from the browser
   * storage.
   * @returns {Promise.<*>}
   */
  BacnetConfigUxManager.prototype.restoreStateForOrd = function (obj) {
    if (obj.$discoveries) {
      this.$discoveries = _.map(obj.$discoveries, function (discovery) {
        return baja.bson.decodeValue(discovery);
      });
      return reloadLearnTable(this);
    } else {
      return Promise.resolve();
    }
  };

  return (BacnetConfigUxManager);
});
