/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/util/PropertyInfo
 */
define(['lex!bacnet',
        'underscore',
        'nmodule/bacnet/rc/util/BacnetConst'], function (
        lexs,
        _,
        BacnetConst) {

  'use strict';

  var bacnetLex = lexs[0],

      ASN_TYPE_MAP = {},
      ASN_TYPE_TO_NAME_MAP = {};

  ASN_TYPE_MAP[BacnetConst.ASN_NULL]              = 'bacnet:BacnetNull';
  ASN_TYPE_MAP[BacnetConst.ASN_BOOLEAN]           = 'baja:Boolean';
  ASN_TYPE_MAP[BacnetConst.ASN_UNSIGNED]          = 'bacnet:BacnetUnsigned';
  ASN_TYPE_MAP[BacnetConst.ASN_INTEGER]           = 'baja:Integer';
  ASN_TYPE_MAP[BacnetConst.ASN_REAL]              = 'baja:Float';
  ASN_TYPE_MAP[BacnetConst.ASN_DOUBLE]            = 'baja:Double';
  ASN_TYPE_MAP[BacnetConst.ASN_OCTET_STRING]      = 'bacnet:BacnetOctetString';
  ASN_TYPE_MAP[BacnetConst.ASN_CHARACTER_STRING]  = 'baja:String';
  ASN_TYPE_MAP[BacnetConst.ASN_BIT_STRING]        = 'bacnet:BacnetBitString';
  ASN_TYPE_MAP[BacnetConst.ASN_DATE]              = 'bacnet:BacnetDate';
  ASN_TYPE_MAP[BacnetConst.ASN_TIME]              = 'bacnet:BacnetTime';
  ASN_TYPE_MAP[BacnetConst.ASN_OBJECT_IDENTIFIER] = 'bacnet:BacnetObjectIdentifier';
  ASN_TYPE_MAP[BacnetConst.ASN_ANY]               = 'bacnet:BacnetAny';

  function lex (key) { return bacnetLex.get(key); }

  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_NULL]                = lex('asn.null');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_BOOLEAN]             = lex('asn.boolean');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_UNSIGNED]            = lex('asn.unsigned');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_INTEGER]             = lex('asn.integer');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_REAL]                = lex('asn.real');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_DOUBLE]              = lex('asn.double');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_OCTET_STRING]        = lex('asn.octetString');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_CHARACTER_STRING]    = lex('asn.characterString');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_BIT_STRING]          = lex('asn.bitString');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_ENUMERATED]          = lex('asn.enumerated');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_DATE]                = lex('asn.date');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_TIME]                = lex('asn.time');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_OBJECT_IDENTIFIER]   = lex('asn.objectId');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASHRAE_RESERVED_13]      = lex('asn.reserved13');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASHRAE_RESERVED_14]      = lex('asn.reserved14');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASHRAE_RESERVED_15]      = lex('asn.reserved15');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_CONSTRUCTED_DATA]    = lex('asn.constructed');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_BACNET_ARRAY]        = lex('asn.array');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_BACNET_LIST]         = lex('asn.list');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_ANY]                 = lex('asn.any');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_CHOICE]              = lex('asn.choice');
  ASN_TYPE_TO_NAME_MAP[BacnetConst.ASN_UNKNOWN_PROPRIETARY] = lex('asn.unknown');

  function attr (elem, name, def) {
    var attr = elem.getAttribute(name);
    return typeof attr === 'string' ? attr : def;
  }

  function parseBoolean (attr) {
    return String(attr).toLowerCase() === 'true';
  }

  ////////////////////////////////////////////////////////////////
  // Construction
  ////////////////////////////////////////////////////////////////

  /**
   * Constructor. Sets the basic properties from the parameters.
   *
   * @class
   * @alias module:nmodule/bacnet/rc/util/PropertyInfo
   */
  var PropertyInfo = function PropertyInfo (param) {
    if (param) {
      if (param.name) { this.$name = param.name; }
      if (param.id !== undefined) { this.$id = param.id; }
      if (param.asnType !== undefined) { this.$asnType = param.asnType; }
    }
  };

  PropertyInfo.prototype.constructor = PropertyInfo;

  /**
   * Make a `PropertyInfo` initialized from the attributes of an XML element.
   * This is used when parsing the objectTypes.xml file.
   *
   * @static
   * @private
   *
   * @param {Object} elem - an XML DOM element read from the file.
   * @returns {module:nmodule/bacnet/rc/util/PropertyInfo}
   */
  PropertyInfo.$makeFromXmlElement = function (elem) {
    var info = new PropertyInfo(),
        asnAttr = attr(elem, 'a');

    info.$id = parseInt(attr(elem, 'i'));
    info.$name = attr(elem, 'n');
    info.$asnType = asnAttr ? parseInt(asnAttr) : BacnetConst.ASN_UNKNOWN_PROPRIETARY;
    info.$type = ASN_TYPE_MAP[info.$asnType];
    info.$facet = parseBoolean(attr(elem, 'f', 'false'));
    info.$facetControl = attr(elem, 'c', 'no');

    if (info.isEnum()) {
      info.$extensible = parseBoolean(attr(elem, 'e', 'false'));
      info.$type = attr(elem, 't');
      info.$facetControl = "enum";
    }
    else if (info.isBitString()) {
      info.$bs = attr(elem, 'b');
    }
    else if (info.isConstructed() || info.isList() || info.isChoice()) {
      info.$type = attr(elem, 't');
    }
    else if (info.isArray()) {
      info.$type = attr(elem, 't');
      info.$size = attr(elem, 's', '-1');
    }

    return info;
  };

  /**
   * Restore the properties on a new instance from a serialized object.
   * @param {Object} props - the deserialized properties for a `PropertyInfo`.
   */
  PropertyInfo.$makeFromCachedProperties = function (o) {
    return _.extend(new PropertyInfo(), o);
  };

////////////////////////////////////////////////////////////////
// Access
////////////////////////////////////////////////////////////////

  PropertyInfo.prototype.getName = function () {
    return this.$name;
  };

  PropertyInfo.prototype.getId = function () {
    return this.$id;
  };

  PropertyInfo.prototype.getType = function () {
    return this.$type;
  };

  PropertyInfo.prototype.getAsnType = function () {
    return this.$asnType;
  };

  PropertyInfo.prototype.isFacet = function () {
    return !!this.$facet;
  };

  PropertyInfo.prototype.getFacetControl = function () {
    return this.$facetControl;
  };

  PropertyInfo.prototype.getDataType = function () {
    return ASN_TYPE_TO_NAME_MAP[String(this.$asnType)] || '???';
  };

  PropertyInfo.prototype.isExtensible = function () {
    if (!this.isEnum()) { throw new Error('enum property required'); }
    return this.$extensible;
  };

  PropertyInfo.prototype.isEnum = function () {
    return this.$asnType === BacnetConst.ASN_ENUMERATED;
  };

  PropertyInfo.prototype.isBitString = function () {
    return this.$asnType === BacnetConst.ASN_BIT_STRING;
  };

  PropertyInfo.prototype.isConstructed = function () {
    return this.$asnType === BacnetConst.ASN_CONSTRUCTED_DATA;
  };

  PropertyInfo.prototype.isArray = function () {
    return this.$asnType === BacnetConst.ASN_BACNET_ARRAY;
  };

  PropertyInfo.prototype.isList = function () {
    return this.$asnType === BacnetConst.ASN_BACNET_LIST;
  };

  PropertyInfo.prototype.isChoice = function () {
    return this.$asnType === BacnetConst.ASN_CHOICE;
  };

  PropertyInfo.prototype.isAws = function () {
    return (this.$type) && (this.$type.indexOf('Aws') >= 0);
  };

  return PropertyInfo;
});
