/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/* jshint browser : true */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/DiscoveryNetworksEditor
 */
define(['baja!',
        'lex!bacnet',
        'jquery',
        'underscore',
        'nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks',
        'nmodule/webEditors/rc/fe/baja/BaseEditor',
        'hbs!nmodule/bacnet/rc/fe/templates/DiscoveryNetworksEditor',
        'css!nmodule/bacnet/rc/bacnet'], function (
        baja,
        lexs,
        $,
        _,
        DiscoveryNetworks,
        BaseEditor,
        tplDiscoveryNetworksEditor) {

  'use strict';

  var bacnetLex = lexs[0],
      NETWORK_NUMBER_COLUMN_COUNT = 5;

  /**
   * Return the jquery object for 'select all' button.
   *
   * @param {JQuery} dom - the editor's jquery dom object
   * @returns {JQuery}
   */
  function getSelectAllElem (dom) {
    return dom.find('.discovery-networks-select-all');
  }

  /**
   * Return the jquery object for 'clear all' button.
   *
   * @param {JQuery} dom - the editor's jquery dom object
   * @returns {JQuery}
   */
  function getClearAllElem (dom) {
    return dom.find('.discovery-networks-clear-all');
  }

  /**
   * Return the jquery objects for the input check boxes corresponding to the
   * network numbers.
   *
   * @param {JQuery} dom - the editor's jquery dom object
   * @returns {JQuery}
   */
  function getNetworkNumberElems (dom) {
    return dom.find('.discovery-network-cb');
  }

  /**
   * Return the jquery object for the 'Send Global' input check box.
   *
   * @param {JQuery} dom - the editor's jquery dom object
   * @returns {JQuery}
   */
  function getSendGlobalElem (dom) {
    return dom.find('.discovery-networks-send-global');
  }
  
  /**
   * Field editor for choosing the network numbers during a BACnet network discovery, or
   * a 'who has' command.
   * 
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/bacnet/rc/fe/DiscoveryNetworksEditor
   */
  var DiscoveryNetworksEditor = function DiscoveryNetworksEditor (params) {
    BaseEditor.call(this, _.extend({
      keyName: 'DiscoveryNetworksEditor',
      moduleName: 'bacnet'
    }, params));
  };
  
  DiscoveryNetworksEditor.prototype = Object.create(BaseEditor.prototype);
  DiscoveryNetworksEditor.prototype.constructor = DiscoveryNetworksEditor;

  /**
   * Load the template and create the buttons and the 'send global' check
   * box. Set up the event handlers.
   *
   * @param {JQuery} dom
   */
  DiscoveryNetworksEditor.prototype.doInitialize = function (dom) {
    var that = this;

    dom.html(tplDiscoveryNetworksEditor({
      sendGlobalId: this.generateId(),
      sendGlobal: bacnetLex.get('discoveryNetworks.global'),
      selectAll: bacnetLex.get('discoveryNetworks.selectAll'),
      clearAll: bacnetLex.get('discoveryNetworks.clearAll')
    }));

    getSelectAllElem(dom).on('click', function () {
      getNetworkNumberElems(dom).prop('checked', true);
      that.setModified(true);
    });

    getClearAllElem(dom).on('click', function () {
      getSendGlobalElem(dom).prop('checked', false);
      getNetworkNumberElems(dom).prop('checked', false);
      that.setModified(true);
    });

    dom.on('change', 'input:checkbox', function () {
      that.setModified(true);
    });
  };

  /**
   * Load the `DiscoveryNetworks` value. This will create a check box for
   * each of the network numbers, each will start initially selected.
   *
   * @param {module:nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks} value
   */
  DiscoveryNetworksEditor.prototype.doLoad = function (value) {
    var that = this,
        jq = that.jq(),
        nets = value.getNetworks(),
        tbody = jq.find('.discovery-networks-table tbody').first(),
        tr;

    _.each(nets, function (num, i) {
      var id = that.generateId(),
          td = $('<td>'),
          input,
          label;

      // For each network number, create a check box with the number as
      // the label.

      input = $('<input type="checkbox" class="discovery-network-cb">')
                .prop('id', id)
                .prop('checked', true)
                .data('net-number', num);

      label = $('<label>' + num + '</label>')
                .prop('for', id);

      // Arrange the checkboxes into a number of columns. Start a new
      // table row when we have reached the maximum number of cells in
      // the current row.

      if (i % NETWORK_NUMBER_COLUMN_COUNT === 0) {
        tr = $('<tr>');
        tbody.append(tr);
      }

      td.append(input).append(label);
      tr.append(td);
    });


    if (value.isAllNetworks()) {
      getSendGlobalElem(jq).prop('checked', true);
    }
  };

  /**
   * Read the state of the input check boxes and return a new `DiscoveryNetworks` instance.
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/DiscoveryNetworks}
   */
  DiscoveryNetworksEditor.prototype.doRead = function () {
    var jq = this.jq(),
        nets = [],
        all = getSendGlobalElem(jq).prop('checked');

    getNetworkNumberElems(jq).each(function (index, elem) {
      var input = $(elem);
      if (input.prop('checked')) {
        nets.push(input.data('net-number'));
      }
    });

    return DiscoveryNetworks.make(all, nets);
  };

  /**
   * Enable or disable the UI elements.
   * @param {Boolean} enabled
   */
  DiscoveryNetworksEditor.prototype.doEnabled = function (enabled) {
    var disable = this.isReadonly() || !enabled,
        jq = this.jq();

    getSendGlobalElem(jq).prop('disabled', disable);
    getNetworkNumberElems(jq).prop('disabled', disable);
    getSelectAllElem(jq).prop('disabled', disable);
    getClearAllElem(jq).prop('disabled', disable);
  };

  /**
   * Enable or disable the UI elements.
   * @param {Boolean} readonly
   */
  DiscoveryNetworksEditor.prototype.doReadonly = function (readonly) {
    var disable = !this.isEnabled() || readonly,
        jq = this.jq();

    getSendGlobalElem(jq).prop('disabled', disable);
    getNetworkNumberElems(jq).prop('disable', disable);
    getSelectAllElem(jq).prop('disabled', disable);
    getClearAllElem(jq).prop('disabled', disable);
  };

  /**
   * Clean up the event handlers on the child elements of the widget.
   */
  DiscoveryNetworksEditor.prototype.doDestroy = function () {
    var jq = this.jq();

    getSelectAllElem(jq).off();
    getClearAllElem(jq).off();

    jq.find('input:checkbox').off();
  };

  return DiscoveryNetworksEditor;
});
