/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */
/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetCalendarEntryEditor
 */
define([
  'Promise',
  'baja!',
  'jquery',
  'bajaux/Widget',
  'bajaux/events',
  'bajaux/mixin/batchSaveMixin',
  'nmodule/webEditors/rc/util/htmlUtils',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor',
  'nmodule/webEditors/rc/fe/baja/util/compUtils',
  'nmodule/webEditors/rc/fe/fe',
  'bajaux/util/CommandButtonGroup',
  'bajaux/commands/ToggleCommandGroup',
  'nmodule/bacnet/rc/fe/BacnetWeekNDayEditor',
  'lex!bacnet',
  'hbs!nmodule/bacnet/rc/fe/templates/BacnetCalendarEntryEditor'], function
  (Promise,
   baja,
   $,
   Widget,
   events,
   batchSaveMixin,
   htmlUtils,
   BaseEditor,
   DisplayOnlyEditor,
   compUtils,
   fe,
   CommandButtonGroup,
   ToggleCommandGroup,
   BacnetWeekNDayEditor,
   lexs,
   templateEditor) {

  'use strict';

  var escapeHtml = htmlUtils.escapeHtml,
    bulkCopy = compUtils.bulkCopy,
    lex = lexs[0],
    DATE_CHOICE = 0,
    DATERANGE_CHOICE = 1,
    WEEKNDAY_CHOICE = 2;

  var DESTROY_EVENT = events.DESTROY_EVENT,
    INITIALIZE_EVENT = events.INITIALIZE_EVENT,
    LOAD_EVENT = events.LOAD_EVENT,
    ENABLE_EVENT = events.ENABLE_EVENT,
    DISABLE_EVENT = events.DISABLE_EVENT,
    MODIFY_EVENT = events.MODIFY_EVENT,
    READONLY_EVENT = events.READONLY_EVENT,
    WRITABLE_EVENT = events.WRITABLE_EVENT;

  /**
   * Editor for handling `bacnet:BacnetCalendarEntry` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/bacnet/rc/fe/BacnetCalendarEntryEditor
   */
  var BacnetCalendarEntryEditor = function BacnetCalendarEntryEditor(params) {
    var that = this;
    BaseEditor.apply(this, arguments);
    if (that.getFormFactor() === Widget.formfactor.mini) {
      DisplayOnlyEditor.$mixin(this);
    }
    else {
      batchSaveMixin(this);
      that.$toggleCmdGrp = new ToggleCommandGroup({
        onChange: function (value) {
          that.$toggleDisplay(value);
        },
        commands: [
          {
            module: "bacnet",
            lex: "BacnetCE.commands.toggle.date",
            value: DATE_CHOICE
          },
          {
            module: "bacnet",
            lex: "BacnetCE.commands.toggle.dateRange",
            value: DATERANGE_CHOICE
          },
          {
            module: "bacnet",
            lex: "BacnetCE.commands.toggle.weekNDay",
            value: WEEKNDAY_CHOICE
          }
        ]
      });
    }
  };
  BacnetCalendarEntryEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetCalendarEntryEditor.prototype.constructor = BacnetCalendarEntryEditor;

  BacnetCalendarEntryEditor.prototype.valueToString = function (value) {
    var dispVal = "";
    if (baja.hasType(value, 'bacnet:BacnetCalendarEntry')) {
      var encVal = value.getDisplay();
      if (encVal) {
        //encoded as 'BBacnetCalendarEntry:<choice> <choiceEncoding>' - just keep the choiceEncoding
        dispVal = encVal.substring(encVal.indexOf(':') + 3);
      }
      else {
        switch (value.getChoice()) {
          case DATE_CHOICE:
            dispVal = value.getDate().encodeToString();
            break;
          case DATERANGE_CHOICE:
            var dr = value.getDateRange();
            dispVal = dr.getStartDate().encodeToString() + ";" + dr.getEndDate().encodeToString();
            break;
          case WEEKNDAY_CHOICE:
            dispVal = value.getWeekNDay().encodeToString();
            break;
          default:
            break;
        }
      }
    }
    return escapeHtml(dispVal || '');
  };

  BacnetCalendarEntryEditor.prototype.$getToggleCommandGroup = function () {
    return this.$toggleCmdGrp;
  };

  BacnetCalendarEntryEditor.prototype.$getDateToggleCommand = function () {
    return this.$getToggleCommandGroup().get(0);
  };

  BacnetCalendarEntryEditor.prototype.$getDateRangeToggleCommand = function () {
    return this.$getToggleCommandGroup().get(1);
  };

  BacnetCalendarEntryEditor.prototype.$getWeeknDayToggleCommand = function () {
    return this.$getToggleCommandGroup().get(2);
  };

  /**
   *
   * @param {JQuery} dom
   */
  BacnetCalendarEntryEditor.prototype.doInitialize = function (dom) {
    var that = this;
    //Disable events from bubbling up
    dom.on([
      DESTROY_EVENT,
      INITIALIZE_EVENT,
      LOAD_EVENT,
      ENABLE_EVENT,
      DISABLE_EVENT,
      READONLY_EVENT,
      WRITABLE_EVENT
    ].join(' '), '.editor', false);

    dom.on(MODIFY_EVENT, '.bajaux-initialized', function () {
      that.setModified(true);
      return false;
    });

    dom.html(templateEditor({
      labels: {
        date: lex.get('BacnetCE.editor.label.date'),
        startDate: lex.get('BacnetCE.editor.label.startDate'),
        endDate: lex.get('BacnetCE.editor.label.endDate')
      }
    }));

    return Promise.join(
      fe.buildFor({
        dom: $('.toggleCommandGrp', dom),
        type: CommandButtonGroup,
        value: that.$getToggleCommandGroup(),
        initializeParams: {
          toggleGroup: true
        },
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$("bacnet:BacnetDate"),
        dom: that.jq().find('.bce-dt'),
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$("bacnet:BacnetDate"),
        dom: that.jq().find('.bce-dr-ds'),
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$("bacnet:BacnetDate"),
        dom: that.jq().find('.bce-dr-de'),
        formFactor: 'mini'
      }),
      fe.buildFor({
        value: baja.$('bacnet:BacnetOctetString', [0, 0, 0]),
        type: BacnetWeekNDayEditor,
        dom: that.jq().find('.bce-wad'),
        formFactor: 'mini'
      })
    );
  };

  /**
   * @param {baja.Simple} value the `bacnet:BacnetCalendarEntry` value to load
   */
  BacnetCalendarEntryEditor.prototype.doLoad = function (value) {
    var that = this;
    var choice = value && value.getChoice();
    that.$toggleDisplay(choice);
    switch (choice) {
      case DATE_CHOICE:
        that.$getDateToggleCommand().setSelected(true);
        return that.$loadDate(value.getDate());
      case DATERANGE_CHOICE:
        that.$getDateRangeToggleCommand().setSelected(true);
        return that.$loadDateRange(value.getDateRange());
      case WEEKNDAY_CHOICE:
        that.$getWeeknDayToggleCommand().setSelected(true);
        return that.$loadWeeknDay(value.getWeekNDay());
      default:
        break;
    }
  };

  BacnetCalendarEntryEditor.prototype.$loadDate = function (dateValue) {
    return this.jq().find('.bce-dt').data('widget').load(dateValue || baja.$("bacnet:BacnetDate"));
  };

  BacnetCalendarEntryEditor.prototype.$loadDateRange = function (dateRangeValue) {
    return Promise.join(
      this.jq().find('.bce-dr-ds').data('widget').load(dateRangeValue.getStartDate() || baja.$("bacnet:BacnetDate")),
      this.jq().find('.bce-dr-de').data('widget').load(dateRangeValue.getEndDate() || baja.$("bacnet:BacnetDate"))
    );
  };

  BacnetCalendarEntryEditor.prototype.$loadWeeknDay = function (weeknDayValue) {
    return this.jq().find('.bce-wad').data('widget')
      .load(baja.$('bacnet:BacnetOctetString', weeknDayValue.getBytes()) || baja.$('bacnet:BacnetOctetString', [0, 0, 0]));
  };

  BacnetCalendarEntryEditor.prototype.doRead = function () {
    var that = this;
    var choice = parseInt(that.$getToggleCommandGroup().getSelectedValue(), 10);
    return that.getChildEditors().readAll()
      .spread(function (date, startDt, endDt, weekAndDay) {
        var ce = baja.$('bacnet:BacnetCalendarEntry');
        switch (choice) {
          case DATE_CHOICE:
            ce.setDate(date);
            ce.setChoice(DATE_CHOICE);
            break;
          case DATERANGE_CHOICE:
            ce.getDateRange().setStartDate(startDt);
            ce.getDateRange().setEndDate(endDt);
            ce.setChoice(DATERANGE_CHOICE);
            break;
          case WEEKNDAY_CHOICE:
            ce.setWeekNDay(weekAndDay);
            ce.setChoice(WEEKNDAY_CHOICE);
            break;
        }
        return ce;
      });
  };

  BacnetCalendarEntryEditor.prototype.doSave = function (readValue, params) {
    var batch = params && params.batch,
        value = this.value(),
        progressCallback = params && params.progressCallback,
        source = readValue, target = value, slots,
        prom;

    switch (readValue.get('choice').valueOf()) {
      case DATE_CHOICE:
        slots = [ 'date' ];
        break;
      case DATERANGE_CHOICE:
        source = readValue.get('dateRange');
        target = value.get('dateRange');
        slots = [ 'startDate', 'endDate' ];
        break;
      case WEEKNDAY_CHOICE:
        slots = [ 'weekNDay' ];
        break;
    }

    prom = Promise.join(
      bulkCopy(readValue, value, [ 'choice' ], batch),
      bulkCopy(source, target, slots, batch)
    );

    if (progressCallback) { progressCallback(batchSaveMixin.COMMIT_READY); }

    return prom;
  };

  /**
   * change the display state based on the calendar entry choice
   * @private
   * @param {Number} choice - The calendar entry value type [0=date, 1=date range, 2=week and day]
   */
  BacnetCalendarEntryEditor.prototype.$toggleDisplay = function (choice) {
    var dateCtrl = this.jq().find(".bce-date");
    var drCtrl = this.jq().find(".bce-dr");
    var wndCtrl = this.jq().find(".bce-wad");
    switch (choice) {
      case DATE_CHOICE:
        drCtrl.hide();
        wndCtrl.hide();
        dateCtrl.show();
        break;
      case DATERANGE_CHOICE:
        dateCtrl.hide();
        wndCtrl.hide();
        drCtrl.show();
        break;
      case WEEKNDAY_CHOICE:
        dateCtrl.hide();
        drCtrl.hide();
        wndCtrl.show();
        break;
      default:
        break;
    }
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  BacnetCalendarEntryEditor.prototype.doReadonly = function (readonly) {
    return this.getChildWidgets().setAllReadonly(readonly);
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  BacnetCalendarEntryEditor.prototype.doEnabled = function (enabled) {
    return this.getChildWidgets().setAllEnabled(enabled);
  };

  BacnetCalendarEntryEditor.prototype.doDestroy = function () {
    return this.getChildWidgets().destroyAll();
  };

  return BacnetCalendarEntryEditor;
});
