/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Raymond A. Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/fe/BacnetBitStringEditor
 */
define([
  'baja!',
  'bajaux/Widget',
  'jquery',
  'underscore',
  'nmodule/webEditors/rc/fe/baja/BaseEditor',
  'nmodule/webEditors/rc/fe/baja/DisplayOnlyEditor',
  'nmodule/bacnet/rc/baja/datatypes/BacnetBitString',
  'nmodule/bacnet/rc/baja/util/BacnetBitStringUtil',
  'hbs!nmodule/bacnet/rc/fe/templates/BacnetBitStringEditor',
  'css!nmodule/bacnet/rc/bacnet'
  ], function (
   baja,
   Widget,
   $,
   _,
   BaseEditor,
   DisplayOnlyEditor,
   BacnetBitString,
   BacnetBitStringUtil,
   tplBacnetBitStringEditor) {

  'use strict';

  ////////////////////////////////////////////////////////////////
  // BacnetBitStringEditor
  ////////////////////////////////////////////////////////////////

  /**
   * Editor for handling 'bacnet:BacnetBitString' values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/baja/rc/fe/BacnetBitStringEditor
   */
  var BacnetBitStringEditor = function BacnetBitStringEditor(params) {
    var that = this;
    that.$bitIds = [];

    BaseEditor.call(that, $.extend({
      keyName: 'BacnetBitStringEditor'
    }, params));

    if (that.getFormFactor() === Widget.formfactor.mini) {
      that.$makeDisplayOnly();
    }

  };
  BacnetBitStringEditor.prototype = Object.create(BaseEditor.prototype);
  BacnetBitStringEditor.prototype.constructor = BacnetBitStringEditor;

  BacnetBitStringEditor.prototype.$makeDisplayOnly = function () {
    DisplayOnlyEditor.$mixin(this);
  };

  /**
   * Create the DOM structure for housing the inputs.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  BacnetBitStringEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html(tplBacnetBitStringEditor({ }));
    dom.on('change', 'input', function () {
      that.setModified(true);
    });
  };


    /**
     * Loads the BacnetBitString into the editor, creating a table row for each
     * bit in the BitString.
     *
     * Each row in the editor has a checkbox with a label.  The label
     * can be set by a facet bit{n}=label; if that's not set, then if the bsName
     * facet is set to a known BacnetBitString type then the bit field names
     * are taken from module.lexicon.
     *
     * @param {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString} value
     */
    BacnetBitStringEditor.prototype.doLoad = function (value) {
      var that = this,
          props = this.properties(),
          numBits = BacnetBitStringUtil.getNumBits(value, props.toValueMap()),
          tbody = that.jq().children('table').children('tbody');

      that.$numBits = numBits;

      // Iterate through the bits and create a checkbox with a label for it
      _.each(_.range(numBits), function (b) {
        var id = that.generateId(),
            input = $('<input type="checkbox">')
              .prop('id', id)
              .prop('checked', value.isBitSet(b)),
            label = $('<label></label>')
              .prop('for', id)
              .text(BacnetBitStringUtil.getBitLabel(value,
                    props.toValueMap(), b));
        that.$bitIds[b] = id;
        tbody.append($('<tr>')
            .append($('<td>')
              .append(input.add(label))
            )
          );
      });
    };

  /**
   * Read the editor data into a `bacnet:BacnetBitString`
   *
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString}
   */
  BacnetBitStringEditor.prototype.doRead = function () {
    var that = this,
        bitString = '',
        inputs = that.jq().children('table').find('input');

    inputs.each(function (bitNumber) {
      var checked = this.checked ? '1' : '0';
      bitString= bitString + checked;
    });
    return BacnetBitString.make(bitString);
  };

  /**
   * When in mini/display-only mode, display the bit field.
   *
   * @param {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString} value
   * @returns {String}
   */
  BacnetBitStringEditor.prototype.valueToString = function (value) {
    return String(value);
  };

  /**
   * Set a bit using the editor.
   *
   * This will check or uncheck checkbox for the specified bit.
   *
   * @param {Number} bitNumber - Bit index to set
   * @param {Boolean|Number} [value=true] - Value to set the bit; if an integer is
   *        specified, only the values 1 or 0 are expected.
   * @param {Boolean} [trigger=false] - If true then trigger a change event.
   */
  BacnetBitStringEditor.prototype.setBit = function (bitNumber, value, trigger) {
    if (value === undefined) {
      value = true;
    }
    var input = this.jq().find('#' + this.$getBitId(bitNumber));
    var prop = input.prop('checked', value ? true : false);
    if(trigger) {
      prop.trigger('change');
    }
  };

  /**
   * Get the HTML element Id for the specified bit number's input element.
   *
   * @private
   * @param {Number} bitNumber
   * @returns {String}
   */
  BacnetBitStringEditor.prototype.$getBitId = function (bitNumber) {
      return this.$bitIds[bitNumber];
  };

  /**
   * Get the HTML label element for the specified bit number's input element.
   *
   * @param {Number} bitNumber
   * @returns {JQuery}
   */
  BacnetBitStringEditor.prototype.getLabelElement = function (bitNumber) {
    return this.jq().children('table').find('label[for="' +
        this.$getBitId(bitNumber) + '"]');
  };

  /**
   * Set all child editors enabled/disabled.
   *
   * @param {Boolean} enabled
   */
  BacnetBitStringEditor.prototype.doEnabled = function (enabled) {
    this.jq().children('table').find('input')
      .prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Set all child editors readonly/writable.
   *
   * @param {Boolean} readonly
   */
  BacnetBitStringEditor.prototype.doReadonly = function (readonly) {
    this.jq().children('table').find('input')
      .prop('disabled', !this.isEnabled() || readonly);
  };

   return BacnetBitStringEditor;
});
