/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/**
 * API Status: **Private**
 * @module nmodule/bacnet/rc/baja/enums/BacnetPropertyIdentifier
 */
define(['baja!',
        'nmodule/bacnet/rc/util/BacnetConst'], function (
        baja,
        BacnetConst) {

  'use strict';

  var FrozenEnum = baja.FrozenEnum,

      INVALID_OR_UNSPECIFIED_ID = 'Invalid ID',
      MAX_RESERVED = 511,
      MAX_ID = 4194303;

  var BacnetPropertyIdentifier = function BacnetPropertyIdentifier() {
    FrozenEnum.apply(this, arguments);
  };

  BacnetPropertyIdentifier.prototype = Object.create(FrozenEnum.prototype);
  BacnetPropertyIdentifier.prototype.constructor = BacnetPropertyIdentifier;

  BacnetPropertyIdentifier.make = function () {
    var def = BacnetPropertyIdentifier.DEFAULT;
    return def.make.apply(def, arguments);
  };

  BacnetPropertyIdentifier.DEFAULT = new BacnetPropertyIdentifier();

  function maxAshrae () {
    var ordinals = BacnetPropertyIdentifier.DEFAULT.getRange().getOrdinals();
    return ordinals[ordinals.length-1];
  }

  /**
   * Is the given id a proprietary extension?
   *
   * @static
   * @param {number} id - the property id number
   * @returns {boolean} true if this is a proprietary extension.
   */
  BacnetPropertyIdentifier.isProprietary = function (id) {
    return (id > MAX_RESERVED) && (id <= MAX_ID);
  };

  /**
   * Is the given id an ASHRAE extension?
   *
   * @static
   * @param {number} id - the property id number
   * @returns {boolean} true if the id is an ASHRAE extension.
   */
  BacnetPropertyIdentifier.isAshrae = function (id) {
    return (id > maxAshrae()) && (id <= MAX_RESERVED);
  };

  /**
   * Is the given id valid for this enumeration?
   *
   * @static
   * @param {number} id - the property id number
   * @returns {boolean} true if the id is within the allowed range.
   */
  BacnetPropertyIdentifier.isValid = function (id) {
    return id <= MAX_ID;
  };

  /**
   * Is the given id part of the predefined (fixed) range?
   *
   * @static
   * @param {number} id - the property id number
   * @returns {boolean} true if this id is in the fixed range.
   */
  BacnetPropertyIdentifier.isFixed = function (id) {
    return id <= maxAshrae();
  };

  /**
   * Get a string tag for the given ordinal value.
   *
   * @static
   * @param {number} id - the property id number
   * @returns {String} a tag for the ordinal, if it is known, or constructed from the standard prefixes.
   */
  BacnetPropertyIdentifier.tagForId = function (id) {
    if (BacnetPropertyIdentifier.DEFAULT.getRange().isOrdinal(id)) {
      return BacnetPropertyIdentifier.DEFAULT.getRange().getTag(id);
    }
    else if (BacnetPropertyIdentifier.isAshrae(id)) {
      return BacnetConst.ASHRAE_PREFIX + id;
    }
    else if (BacnetPropertyIdentifier.isProprietary(id)) {
      return BacnetConst.PROPRIETARY_PREFIX + id;
    }
    else {
      return INVALID_OR_UNSPECIFIED_ID;
    }
  };

  /**
   * Get an ordinal number from the given string tag.
   *
   * @static
   * @param {String} tag - a string containing the tag name for the property.
   * @returns {Number} - the ordinal number of the given tag, parsing the standard prefixes if necessary,
   * or -1 if the id could not be parsed from the tag string.
   */
  BacnetPropertyIdentifier.idForTag = function (tag) {
    if (BacnetPropertyIdentifier.DEFAULT.getRange().isTag(tag)) {
      return BacnetPropertyIdentifier.DEFAULT.make(tag).getOrdinal();
    }
    else if (tag.indexOf(BacnetConst.ASHRAE_PREFIX) === 0) {
      return parseInt(tag.substring(BacnetConst.ASHRAE_PREFIX.length), 10);
    }
    else if (tag.indexOf(BacnetConst.PROPRIETARY_PREFIX) === 0) {
      return parseInt(tag.substring(BacnetConst.PROPRIETARY_PREFIX.length), 10);
    }

    return -1;
  };

  return BacnetPropertyIdentifier;
});
