/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Raymond A. Richards
 */

 /**
  * API Status: **Private**
  * @module nmodule/bacnet/rc/baja/datatypes/BacnetBitString
  */
  define(['baja!'], function (
      baja) {

  'use strict';

  var Simple = baja.Simple;

  /**
   * BajaScript representation of a `bacnet:BacnetBitString` value.
   *
   * @class
   * @extends {baja.Simple}
   * @alias {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString}
   */
  var BacnetBitString = function BacnetBitString() {
    Simple.apply(this, arguments);
  };
  BacnetBitString.prototype = Object.create(Simple.prototype);
  BacnetBitString.prototype.constructor = BacnetBitString;

  /**
   * Create a new `bacnet:BacnetBitString` instance from the arguments.
   *
   * @param {String} bits - String representation of the bits in the bit string
   * @param {String} tags
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString}
   */
  BacnetBitString.make = function (bits, tags) {
    var bbs = new BacnetBitString();
    bbs.$bits = bits === undefined ? '' : bits;
    bbs.$tags = tags === undefined ? '' : tags;

    return bbs;
  };

  /**
   * Create a new `BacnetBitString` instance from the arguments.
   *
   * @see .make
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString}
   */
  BacnetBitString.prototype.make = function () {
    return BacnetBitString.make.apply(BacnetBitString, arguments);
  };

  /**
   * Decode an `BacnetBitString` from a string.
   *
   * @param {String} str - Colon separated string; the left portion is a string
   *      representation of the bits, and the right portion is a string
   *      representation of the facets /tags.
   * @returns {module:nmodule/bacnet/rc/baja/datatypes/BacnetBitString}
   */
  BacnetBitString.prototype.decodeFromString = function (str) {
    if (!str || str === '') {
      return BacnetBitString.DEFAULT;
    }
    var parts = str.split(':');
    return BacnetBitString.make.apply(null, parts);
  };

  /**
   * Encode this `BacnetBitString` to a string.
   *
   * This is the opposite of `decodeFromString`.
   *
   * @returns {String}
   */
  BacnetBitString.prototype.encodeToString = function () {
    return this.$bits + ':' + this.$tags;
  };

  /**
   * Returns a simple string representation of the `BacnetBitString`.
   *
   * @returns {String}
   */
  BacnetBitString.prototype.toString = function (cx) {
    return this.$bits;
  };

  /**
   * Gets the number of bits for this `BacnetBitString`.
   *
   * @returns {String}
   */
  BacnetBitString.prototype.getNumBits = function () {
    return this.$bits.length;
  };

  /**
   * Checks if the specified bit is set to 1.
   *
   * @param {Number} bitNumber - Indexed bit number
   * @returns {Boolean}
   */
  BacnetBitString.prototype.isBitSet = function (bitNumber) {
    return this.$bits.charAt(bitNumber) === '1';
  };

  BacnetBitString.DEFAULT = BacnetBitString.make();

  baja.registerType('bacnet:BacnetBitString', function () {
    return BacnetBitString;
  });

  return BacnetBitString;
});
