/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/util/ExpirationEditor
 */
define(['baja!', 'lex!webEditors', 'Promise', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'hbs!nmodule/webEditors/rc/wb/util/template/ExpirationEditor'], function (baja, lexs, Promise, fe, BaseEditor, tplExpirationEditor) {
  'use strict';

  var webEditorsLex = lexs[0];

  //TODO: NCCB-10255
  function isDefault(absTime) {
    return absTime.getDate().getYear() <= 1970;
  }
  function endOfToday() {
    return baja.AbsTime.make({
      date: baja.Date.today(),
      time: baja.Time.make({
        hour: 23,
        min: 59,
        sec: 59
      }),
      offset: baja.AbsTime.now().getOffset()
    });
  }

  /**
   * Editor for setting the expiration date on a User.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/util/ExpirationEditor
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   */
  var ExpirationEditor = function ExpirationEditor() {
    BaseEditor.apply(this, arguments);
  };
  ExpirationEditor.prototype = Object.create(BaseEditor.prototype);
  ExpirationEditor.prototype.constructor = ExpirationEditor;

  /**
   * Get the radio button for "never expires".
   *
   * @private
   * @returns {jQuery}
   */
  ExpirationEditor.prototype.$getNeverButton = function () {
    return this.jq().find('input[value=never]');
  };

  /**
   * Get the radio button for "expires on".
   *
   * @private
   * @returns {jQuery}
   */
  ExpirationEditor.prototype.$getExpiresButton = function () {
    return this.jq().find('input[value=expires]');
  };

  /**
   * Get the element where the AbsTime editor lives.
   *
   * @private
   * @returns {jQuery}
   */
  ExpirationEditor.prototype.$getAbsTimeElement = function () {
    return this.jq().children('.absTime');
  };

  /**
   * Return the AbsTimeEditor with the AbsTime loaded.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/AbsTimeEditor}
   */
  ExpirationEditor.prototype.$getAbsTimeEditor = function () {
    return this.$getAbsTimeElement().data('widget');
  };
  ExpirationEditor.prototype.$isNever = function () {
    return this.$getNeverButton().prop('checked');
  };

  /**
   * Update the enabled status of the AbsTime editor based on whether
   * "never expires" or "expires on" is checked.
   *
   * @private
   * @returns {Promise} promise to be resolved when the editor is enabled
   * or disabled
   */
  ExpirationEditor.prototype.$updateFromButtons = function () {
    return this.$getAbsTimeEditor().setEnabled(!this.$isNever());
  };

  /**
   * Enable/disable the editor radio buttons.
   *
   * @private
   * @param {Boolean} enabled
   */
  ExpirationEditor.prototype.$updateButtons = function (enabled) {
    this.$getNeverButton().prop('disabled', !enabled);
    this.$getExpiresButton().prop('disabled', !enabled);
  };

  /**
   * Sets up radio buttons and AbsTime editor. Arms handlers to enable/disable
   * the editor based on which radio button is checked. Adds "ExpirationEditor"
   * class.
   *
   * @param {JQuery} dom
   * @returns {Promise}
   */
  ExpirationEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('ExpirationEditor').html(tplExpirationEditor({
      neverExpires: webEditorsLex.get('ExpirationEditor.neverExpires'),
      expiresOn: webEditorsLex.get('ExpirationEditor.expiresOn'),
      neverId: that.generateId(),
      expiresId: that.generateId(),
      name: that.generateId()
    })).on('change', 'input[type=radio]', function () {
      that.$updateFromButtons();
      that.setModified(true);
      return false;
    });
    return fe.makeFor({
      type: 'baja:AbsTime',
      formFactor: 'mini'
    }).then(function (ed) {
      return ed.initialize(dom.children('.absTime'));
    });
  };

  /**
   * Load the AbsTime into the editor and update the state of the radio buttons
   * appropriately.
   *
   * @param {baja.AbsTime} value
   * @returns {Promise}
   */
  ExpirationEditor.prototype.doLoad = function (value) {
    var that = this,
      ed = that.$getAbsTimeEditor();
    if (isDefault(value)) {
      that.$getNeverButton().prop('checked', true);
      value = endOfToday();
    } else {
      that.$getExpiresButton().prop('checked', true);
    }
    return Promise.all([ed.load(value), that.$updateFromButtons()]);
  };

  /**
   * Return `AbsTime.DEFAULT` if "never expires" is checked, otherwise the
   * currently entered AbsTime value.
   */
  ExpirationEditor.prototype.doRead = function () {
    var isNever = this.$isNever();
    return isNever ? baja.AbsTime.DEFAULT : this.$getAbsTimeEditor().read();
  };

  /**
   * If "never" is checked, load the end of today into the AbsTime editor (for
   * editing if "expires" is checked later).
   *
   * @returns {Promise}
   */
  ExpirationEditor.prototype.doSave = function () {
    if (this.$isNever()) {
      return this.$getAbsTimeEditor().load(endOfToday());
    }
  };

  /**
   * Removes "ExpirationEditor" class and destroys the AbsTime editor.
   *
   * @returns {Promise}
   */
  ExpirationEditor.prototype.doDestroy = function () {
    this.jq().removeClass('ExpirationEditor');
    return this.$getAbsTimeEditor().destroy();
  };

  /**
   * Readonly/writable the radio buttons and AbsTime editor.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  ExpirationEditor.prototype.doReadonly = function (readonly) {
    this.$updateButtons(this.isEnabled() && !readonly);
    return this.$getAbsTimeEditor().setReadonly(readonly);
  };

  /**
   * Enable/disable the radio buttons and AbsTime editor.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  ExpirationEditor.prototype.doEnabled = function (enabled) {
    this.$updateButtons(!this.isReadonly() && enabled);
    return this.$getAbsTimeEditor().setEnabled(enabled);
  };
  return ExpirationEditor;
});
