/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'lex!webEditors', 'dialogs', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/commands/RenameCommand'], function (baja, lexs, dialogs, Promise, _, feDialogs, typeUtils, RenameCommand) {
  'use strict';

  var webEditorsLex = lexs[0];
  var isComponent = typeUtils.isComponent;

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/wb/transfer/transferUtils
   */
  var exports = {};

  /**
   * Perform the function from this module that corresponds to the given
   * transfer op name.
   *
   * @param {string} transferOp
   * @param {Array} subject
   * @returns {Promise}
   */
  exports.doDefault = function (transferOp, subject) {
    var func = exports[transferOp];
    if (func) {
      return func(subject);
    }
    return Promise.reject(new Error('transfer op "' + transferOp + '" not supported'));
  };

  /**
   * Decide whether the transfer op is supported based on usual Niagara
   * rules.
   *
   * - `delete`: subject must be all Components on which you have admin write
   *   privileges.
   * - `rename`: subject must be exactly one parented Component on which you
   *   have admin write privileges. (We do not yet have the ability to do
   *   find-replace for renaming multiple Components as we do in Workbench.)
   *
   * @param {string} transferOp
   * @param {Array} subject
   * @returns {boolean}
   */
  exports.isDefaultEnabled = function (transferOp, subject) {
    switch (transferOp) {
      case 'delete':
        return allComponents(subject) && allAdminWrite(subject);
      case 'rename':
        return subject.length === 1 && allComponents(subject) && allAdminWrite(subject) && !!subject[0].getParent();
    }
  };

  //TODO: these will ultimately need to route through an RPC to allow delegation
  //to the component space.

  /**
   * Confirm for deletion, then delete all components in the subject.
   * @param {Array.<baja.Component>} subject
   * @returns {Promise}
   */
  exports["delete"] = function (subject) {
    if (!allComponents(subject)) {
      return Promise.reject(new Error("Can only delete a collection of all components"));
    }
    return dialogs.showOkCancel({
      title: webEditorsLex.get('commands.delete.confirm.title'),
      content: webEditorsLex.getSafe('commands.delete.confirm.content')
    }).ok(function () {
      return deleteComponents(subject)["catch"](feDialogs.error);
    }).promise();
  };

  /**
   * Prompt for a new slot name and rename the given component.
   * @param {Array.<baja.Component>} subject (must be exactly 1)
   * @returns {Promise}
   */
  exports.rename = function (subject) {
    if (subject.length !== 1) {
      return Promise.reject(new Error('rename() subject must be length 1'));
    }
    if (!allComponents(subject) || !subject[0].getParent()) {
      return Promise.reject(new Error('can only rename parented Components'));
    }
    var comp = subject[0];
    return new RenameCommand(comp.getParent()).invoke({
      slots: [comp.getPropertyInParent()]
    });
  };
  function hasAdminWrite(comp) {
    return comp.getPermissions().hasAdminWrite();
  }
  function allAdminWrite(comps) {
    return _.all(comps, hasAdminWrite);
  }
  function deleteComponents(comps) {
    var batch = new baja.comm.Batch(),
      proms = comps.map(function (comp) {
        var prop = comp.getPropertyInParent();
        return comp.getParent().remove({
          slot: prop,
          batch: batch
        });
      });
    batch.commit();
    return Promise.all(proms);
  }
  function allComponents(comps) {
    return _.all(comps, isComponent);
  }
  return exports;
});
