/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/table/pagination/PaginationModel
 */
define(['Promise', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/util/storageUtil'], function (Promise, tinyevents, storageUtil) {
  'use strict';

  var PAGE_SIZE_KEY = 'pagination.pageSize';

  /**
   * Represents the model backing a paginated Table.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel
   * @mixes tinyevents
   * @param {object} [params]
   * @param {number} [params.rowsPerPage=100] rows per page
   * @param {number} [params.rowCount=-1] row count. -1 indicates unlimited, or
   * unknown, number of total rows.
   * @param {number} [params.currentPage=1] current page
   * @param {boolean} [params.sortingAllowed] PaginationModel does not sortColumns
   * by default, but passing in this parameter can modify this behavior if this
   * parameter is provided.
   */
  var PaginationModel = function PaginationModel(params) {
    this.initialize(params);
    tinyevents(this);
  };

  /**
   * Separate initialization, which will (re)-initialize the pagination model
   * without firing any events.
   *
   * @param {object} [params]
   * @param {number} [params.rowsPerPage=100] rows per page
   * @param {number} [params.rowCount=-1] row count. -1 indicates unlimited, or
   * unknown, number of total rows.
   * @param {number} [params.currentPage=1] current page
   * @param {boolean} [params.sortingAllowed] PaginationModel does not sortColumns
   * by default, but passing in this parameter can modify this behavior if this
   * parameter is provided.
   */
  PaginationModel.prototype.initialize = function (params) {
    params = params || {};
    var rowCount = params.rowCount;
    if (params.sortingAllowed !== undefined) {
      this.$sortingAllowed = params.sortingAllowed;
    }
    this.$rowsPerPage = params.rowsPerPage || PaginationModel.DEFAULT_ROWS_PER_PAGE;
    this.$rowCount = typeof rowCount === 'number' ? rowCount : -1;
    this.$currentPage = params.currentPage || 1;
  };

  /**
   * Given a range of row numbers, the implementer should resolve a TableModel
   * containing the correct rows.
   *
   * Note that during certain operations (such as export), `rowEnd` may be
   * greater than the actual number of available rows. In this case,
   * `makeTableModel()` should not reject but simply resolve a TableModel with
   * as many rows as possible.
   *
   * @abstract
   * @param {number} rowStart (inclusive)
   * @param {number} rowEnd (exclusive)
   * @param {object} [context] optional context
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/model/TableModel>|
   *           module:nmodule/webEditors/rc/wb/table/model/TableModel}
   */
  PaginationModel.prototype.makeTableModel = function (rowStart, rowEnd, context) {
    throw new Error('not implemented');
  };

  /**
   * @returns {number}
   */
  PaginationModel.prototype.getRowsPerPage = function () {
    return this.$rowsPerPage;
  };

  /**
   * @param {number} rowsPerPage
   * @returns {Promise} value is set synchronously; promise resolves after
   * any async handlers do
   */
  PaginationModel.prototype.setRowsPerPage = function (rowsPerPage) {
    return Promise.all(this.emit('changed', 'rowsPerPage', this.$rowsPerPage = rowsPerPage || PaginationModel.DEFAULT_ROWS_PER_PAGE));
  };

  /**
   * @returns {number}
   */
  PaginationModel.prototype.getRowCount = function () {
    return this.$rowCount;
  };

  /**
   * @param {number} rowCount
   * @returns {Promise} value is set synchronously; promise resolves after
   * any async handlers do
   */
  PaginationModel.prototype.setRowCount = function (rowCount) {
    return Promise.all(this.emit('changed', 'rowCount', this.$rowCount = rowCount || 0));
  };

  /**
   * @returns {number}
   */
  PaginationModel.prototype.getCurrentPage = function () {
    return this.$currentPage;
  };

  /**
   * @param {number} currentPage
   * @returns {Promise} value is set synchronously; promise resolves after
   * any async handlers do
   */
  PaginationModel.prototype.setCurrentPage = function (currentPage) {
    var pageCount = this.getPageCount();
    if (currentPage < 1 || pageCount >= 0 && currentPage > pageCount) {
      throw new Error('page out of range');
    }
    return Promise.all(this.emit('changed', 'currentPage', this.$currentPage = currentPage));
  };

  /**
   * @returns {number} number of available pages; -1 if number of pages is
   * unlimited or unknown
   */
  PaginationModel.prototype.getPageCount = function () {
    var rowCount = this.getRowCount();
    if (rowCount < 0) {
      return -1;
    }
    if (rowCount === 0) {
      return 1;
    }
    return Math.ceil(rowCount / this.getRowsPerPage());
  };

  /**
   * @param {number} pageNumber
   * @param {object} [context] optional context
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/model/TableModel>}
   */
  PaginationModel.prototype.resolvePage = function (pageNumber, context) {
    var that = this,
      rowsPerPage = this.getRowsPerPage(),
      rowCount = this.getRowCount(),
      rowStart = (pageNumber - 1) * rowsPerPage,
      rowEnd = rowStart + rowsPerPage;
    if (rowCount !== -1) {
      rowEnd = Math.min(rowEnd, rowCount);
    }
    return Promise.resolve(that.makeTableModel(Math.min(rowStart, rowEnd), rowEnd)).then(function (model) {
      if (!that.isSortingAllowed()) {
        var columns = model.getColumns();
        columns.forEach(function (column) {
          column.setSortable(false);
        });
      }
      return model;
    });
  };

  /**
   * Sets whether the tables should be sortable.
   * @param {boolean} sortable
   */
  PaginationModel.prototype.setSortingAllowed = function (sortable) {
    this.$sortingAllowed = sortable;
  };

  /**
   * Gets whether or not the table is sortable.
   * @returns {boolean}
   */
  PaginationModel.prototype.isSortingAllowed = function () {
    return !!this.$sortingAllowed;
  };

  /**
   * Reads user-configured settings from session storage and applies them.
   * @returns {module:nmodule/webEditors/rc/wb/table/pagination/PaginationModel} this
   */
  PaginationModel.prototype.readFromStorage = function () {
    var storage = storageUtil.getSessionStorage();
    this.$rowsPerPage = storage.getItem(PAGE_SIZE_KEY) || PaginationModel.DEFAULT_ROWS_PER_PAGE;
    return this;
  };
  PaginationModel.prototype.writeToStorage = function () {
    var storage = storageUtil.getSessionStorage();
    storage.setItem(PAGE_SIZE_KEY, this.getRowsPerPage());
  };
  PaginationModel.MAX_ROWS = 0x7fffffff - 1; //allow for safe Integer.parseInt() on Java side

  PaginationModel.DEFAULT_ROWS_PER_PAGE = 20;
  return PaginationModel;
});
