/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @module nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/Attachable', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/table/model/ComponentSource'], function (baja, Promise, _, Attachable, typeUtils, ComponentSource) {
  'use strict';

  var getUniqueSlotName = typeUtils.getUniqueSlotName;
  function toTypeFilter(filterTypes) {
    filterTypes = _.map(filterTypes, String);
    return function (slot) {
      return isVisible(slot) && !!_.find(filterTypes, function (filterType) {
        return slot.getType().is(filterType);
      });
    };
  }
  function isVisible(slot) {
    return !(slot.getFlags() & baja.Flags.HIDDEN);
  }

  /**
   * API Status: **Development**
   *
   * This `ComponentSource` implementation takes a `Component` container and an
   * optional array of Types to filter against. For instance, you'd pass in
   * a `UserService` and retrieve all the `User` slots. It will attach handlers
   * to fire `added` and `removed` events when matching Components are added
   * or removed from the container, or `changed` events when child components
   * have their properties change.
   *
   * Since Baja event handlers will be attached to the input component, ensure
   * that you call `destroy()` when done with the `ComponentSource` to detach
   * these handlers and avoid memory leaks.
   *
   * By default, hidden slots will always be excluded. To include hidden slots,
   * pass a custom filter function.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/table/model/source/ContainerComponentSource
   * @extends module:nmodule/webEditors/rc/wb/table/model/ComponentSource
   *
   * @param {Object|baja.Component} obj params object, or the `Component`
   * container itself if no other params needed
   * @param {baja.Component} obj.container the `Component` container
   * @param {Array.<Type|String>|Function} [obj.filter] if given, only child
   * components matching these Types will be returned from `getComponents` or
   * cause events to be fired. This can also be a function that takes a `Slot` as
   * its first argument and the `Slot`'s value as its second, and should return true
   * or false to include or exclude the value.
   */
  var ContainerComponentSource = function ContainerComponentSource(obj) {
    ComponentSource.apply(this, arguments);
    var that = this,
      container = that.getContainer(),
      filter = obj.filter;
    if (Array.isArray(filter)) {
      filter = toTypeFilter(filter);
    } else if (typeof filter !== 'function') {
      filter = isVisible;
    }
    that.$filter = filter;
    that.$attachable = that.makeAttachable(container);
  };
  ContainerComponentSource.prototype = Object.create(ComponentSource.prototype);
  ContainerComponentSource.prototype.constructor = ContainerComponentSource;

  /**
   * Gets an array of all matching child Components of the container.
   *
   * @returns {Array.<baja.Component>}
   */
  ContainerComponentSource.prototype.getComponents = function () {
    var filter = this.$filter;
    return this.$container.getSlots().properties().is('baja:Complex').filter(function (slot) {
      return filter(slot, this.get(slot));
    }).toValueArray();
  };

  /**
   * Add the (unmounted) components to the container. If names are also given,
   * those names will be used; otherwise default slot names will be generated.
   *
   * If the current container component is not subscribed, an `added` tinyevent
   * will be emitted by this function; otherwise `added` will be emitted by
   * the subscriber attached in the `ContainerComponentSource` constructor.
   *
   * @param {Array.<baja.Component>} comps
   * @param {Array.<String>} [names]
   * @returns {Promise}
   */
  ContainerComponentSource.prototype.addComponents = function (comps, names) {
    var that = this,
      batch = new baja.comm.Batch(),
      container = that.$container,
      adds = _.map(comps, function (comp, i) {
        var slotName = names && names[i] || getUniqueSlotName(container, comp.getType()) + '?';
        return container.add({
          slot: slotName,
          value: comp,
          batch: batch
        });
      });
    batch.commit();
    return Promise.all(adds).then(function (props) {
      //if subscribed, we're already attached for added events
      if (!container.isSubscribed()) {
        var values = _.map(props, function (prop) {
          return container.get(prop);
        });
        that.handleAdded(values);
      }
    });
  };

  /**
   * Remove the given components from the container. If any of the components
   * are not currently children of the current container, no action will be
   * taken for that component.
   *
   * If the current container component is not subscribed, a `removed` tinyevent
   * will be emitted by this function; otherwise `removed` will be emitted by
   * the subscriber attached in the `ContainerComponentSource` constructor.
   *
   * @param {Array.<baja.Component>} comps
   * @returns {Promise}
   */
  ContainerComponentSource.prototype.removeComponents = function (comps) {
    var that = this,
      batch = new baja.comm.Batch(),
      container = that.$container,
      removes = _.map(comps, function (comp) {
        if (comp.getParent() === container) {
          return container.remove({
            slot: comp.getPropertyInParent(),
            batch: batch
          });
        }
      });
    batch.commit();
    return Promise.all(removes).then(function () {
      //if subscribed, we're already attached for added events
      if (!container.isSubscribed()) {
        that.handleRemoved(comps);
      }
    });
  };

  /**
   * Create a new `Attachable` instance for the given component, that will
   * hook up to the various 'handle*' methods. The default implementations
   * of these will emit the appropriate event from this instance.
   *
   * @param {baja.Component} comp - a `Component` to be attached to.
   * @returns {module:nmodule/webEditors/rc/fe/baja/util/Attachable}
   */
  ContainerComponentSource.prototype.makeAttachable = function (comp) {
    var that = this,
      filter = that.$filter,
      att = new Attachable(comp);
    att.attach({
      'added': function added(prop) {
        if (filter(prop, this.get(prop))) {
          that.handleAdded([this.get(prop)]);
        }
      },
      'removed': function removed(prop, value) {
        if (filter(prop, value)) {
          that.handleRemoved([value]);
        }
      },
      'renamed': function renamed(prop, oldName) {
        if (filter(prop, this.get(prop))) {
          that.handleChanged(this.get(prop));
        }
      },
      '.+/**/changed': function __changed(prop) {
        var changed = this,
          changedSlot = prop,
          matchingChild = this;

        // walk up to matching child of the root component to decide which
        // component to fire the event against
        while (matchingChild.getParent() !== comp) {
          matchingChild = matchingChild.getParent();
          changedSlot = changed.getPropertyInParent();
          changed = changed.getParent();
        }
        if (filter(matchingChild.getPropertyInParent(), matchingChild)) {
          that.handleChanged(matchingChild, changedSlot);
        }
      }
    });
    return att;
  };

  /**
   * Emits an `added` event when a new property is added.
   *
   * @param {Array.<baja.Component>} values
   */
  ContainerComponentSource.prototype.handleAdded = function (values) {
    this.emit('added', values);
  };

  /**
   * Emits a `removed` event when a property is removed from the source.
   *
   * @param {Array.<baja.Component>} values
   */
  ContainerComponentSource.prototype.handleRemoved = function (values) {
    this.emit('removed', values);
  };

  /**
   * Emits a `changed` event when a property is changed or renamed.
   *
   * @param {baja.Component} comp
   * @param {baja.Property} prop
   */
  ContainerComponentSource.prototype.handleChanged = function (comp, prop) {
    this.emit('changed', comp, prop);
  };

  /**
   * Clean up Baja event handlers attached to the container Component.
   */
  ContainerComponentSource.prototype.destroy = function () {
    if (this.$attachable) {
      this.$attachable.detach();
      delete this.$attachable;
    }
  };
  return ContainerComponentSource;
});
