/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/table/model/source/ArrayComponentSource
 */
define(['Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/table/model/ComponentSource'], function (Promise, _, typeUtils, ComponentSource) {
  'use strict';

  var isComplex = typeUtils.isComplex,
    isComponent = typeUtils.isComponent;

  /** @param {String} err */
  function reject(err) {
    return Promise.reject(new Error(err));
  }
  function isArrayOfComplexes(arr) {
    if (!_.isArray(arr)) {
      return false;
    }
    for (var i = 0; i < arr.length; i++) {
      if (!isComplex(arr[i])) {
        return false;
      }
    }
    return true;
  }

  /**
   * `ComponentSource` implementation backed by a simple array of `Component`s.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/table/model/source/ArrayComponentSource
   * @extends module:nmodule/webEditors/rc/wb/table/model/ComponentSource
   * @throws {Error} if container is not an array of Components
   */
  var ArrayComponentSource = function ArrayComponentSource() {
    ComponentSource.apply(this, arguments);
    var that = this,
      container = that.getContainer();
    if (!isArrayOfComplexes(container)) {
      throw new Error('array of Components required');
    }
    that.$changeHandler = function (prop) {
      that.emit('changed', this, prop);
    };
    that.$attach(container);
  };
  ArrayComponentSource.prototype = Object.create(ComponentSource.prototype);
  ArrayComponentSource.prototype.constructor = ArrayComponentSource;

  /**
   * Attach 'changed' handler.
   *
   * @private
   * @param {Array.<baja.Component>} comps
   */
  ArrayComponentSource.prototype.$attach = function (comps) {
    var that = this;
    _.each(comps, function (comp) {
      if (isComponent(comp)) {
        comp.attach('changed', that.$changeHandler);
      }
    });
  };

  /**
   * Detach `changed` handler.
   *
   * @private
   * @param {Array.<baja.Component>} comps
   */
  ArrayComponentSource.prototype.$detach = function (comps) {
    var that = this;
    _.each(comps, function (comp) {
      if (isComponent(comp)) {
        comp.detach('changed', that.$changeHandler);
      }
    });
  };

  /**
   * Get the backing array of components.
   *
   * @returns {Array.<baja.Component>}
   */
  ArrayComponentSource.prototype.getComponents = function () {
    return this.getContainer();
  };

  /**
   * Push new components onto the array. Emits an `added` event.
   *
   * @param {Array.<baja.Component>} comps
   * @returns {Promise} promise to be resolved if component successfully
   * added
   */
  ArrayComponentSource.prototype.addComponents = function (comps) {
    for (var i = 0, len = comps.length; i < len; i++) {
      var comp = comps[i];
      if (!isComplex(comp)) {
        return reject('Complex required');
      }
      this.getContainer().push(comp);
      this.$attach([comp]);
    }
    this.emit('added', comps);
    return Promise.resolve();
  };

  /**
   * Removes components from the array.
   *
   * @param {Array.<baja.Component>} comps
   * @returns {Promise} promise to be resolved if components are removed
   */
  ArrayComponentSource.prototype.removeComponents = function (comps) {
    var container = this.getContainer(),
      comp,
      i,
      idx,
      len;
    for (i = 0, len = comps.length; i < len; i++) {
      comp = comps[i];
      if (!isComplex(comp)) {
        return reject('Complex required');
      }
      idx = container.indexOf(comp);
      if (idx >= 0) {
        container.splice(idx, 1);
        this.$detach([comp]);
      } else {
        return reject('given component did not exist in the array');
      }
    }
    this.emit('removed', comps);
    return Promise.resolve();
  };

  /**
   * Remove all `changed` handlers.
   */
  ArrayComponentSource.prototype.destroy = function () {
    this.$detach(this.getContainer());
  };
  return ArrayComponentSource;
});
