/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['underscore', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/transfer/transferUtils'], function (_, mixinUtils, transferUtils) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    MIXIN_NAME = 'transfer',
    DEFAULT_TRANSFER_OPS = ['cut', 'copy', 'paste', 'pasteSpecial', 'duplicate', 'delete', 'rename'];
  /**
   * API Status: **Private**
   *
   * `TransferSupport` adds the ability to perform transfer commands to an
   * object. It is analogous to the functions provided by `BTransferWidget`,
   * but provides the capability to provide a custom set of supported transfer
   * commands instead of having to implement `BTransferWidget`'s hardcoded set.
   *
   * @exports nmodule/webEditors/rc/wb/mixin/TransferSupport
   * @see module:nmodule/webEditors/rc/wb/transfer/transferUtils
   */
  var exports = {
    /**
     * Perform the transfer op. By default, it will delegate to
     * `transferUtils.doDefault`.
     * @param {string} op
     * @param {Array} subject
     * @returns {Promise}
     */
    doTransferOp: function doTransferOp(op, subject) {
      return transferUtils.doDefault(op, subject);
    },
    /**
     * Get the current subject on which to perform the transfer op. For
     * instance, a nav tree would resolve the values of the currently
     * highlighted tree nodes. Each recipient of this mixin must implement it
     * itself.
     *
     * As transfer ops are done as the result of the user interacting with the
     * DOM in some way (e.g. a right-click or keypress), the DOM element
     * that received that event should be provided. (If for some reason the
     * activated element is not known, just use `widget.jq()`.)
     *
     * This function is Promise-optional, so when calling directly, be sure to
     * wrap in `Promise.resolve()` as needed.
     *
     * @param {JQuery} dom
     * @returns {Array|Promise.<Array>}
     */
    getSubject: function getSubject(dom) {
      throw new Error('Not implemented');
    },
    /**
     * Given a subject, the owner can specify the set of transfer ops that are
     * supported for it.
     *
     * @param {Array} subject
     * @returns {Array.<string>}
     */
    getSupportedTransferOps: function getSupportedTransferOps(subject) {
      return DEFAULT_TRANSFER_OPS.slice();
    },
    /**
     * Specify whether the given transfer op is enabled for a particular
     * subject. By default, it will delegate to
     * `transferUtils.isDefaultEnabled`.
     *
     * @param {string} op
     * @param {Array} subject
     * @returns {boolean}
     */
    isTransferOpEnabled: function isTransferOpEnabled(op, subject) {
      return transferUtils.isDefaultEnabled(op, subject);
    },
    /**
     * Return true if the given transfer op is supported for this subject.
     * @param {string} op
     * @param {Array} subject
     * @returns {boolean}
     */
    isTransferOpSupported: function isTransferOpSupported(op, subject) {
      return _.contains(this.getSupportedTransferOps(subject), op);
    }
  };
  function TransferSupport(target) {
    if (!_.isObject(target)) {
      throw new Error('object required');
    }
    applyMixin(target, MIXIN_NAME, exports);
  }
  TransferSupport.DEFAULT_TRANSFER_OPS = DEFAULT_TRANSFER_OPS;
  TransferSupport.toMap = function (widget, subject) {
    var map = {};
    widget.getSupportedTransferOps(subject).forEach(function (op) {
      map[op] = widget.isTransferOpEnabled(op, subject);
    });
    return map;
  };
  return TransferSupport;
});
