/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser: true */
/*global niagara */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/mixin/ComponentHyperlinkColumnMixin
 */
define(['jquery', 'Promise', 'nmodule/webEditors/rc/wb/table/model/Column', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function ($, Promise, Column, mixinUtils) {
  "use strict";

  var applyMixin = mixinUtils.applyMixin,
    MIXIN_NAME = 'HYPERLINK_COLUMN';

  /**
   * Return a session relativized ord for a row's subject or null if the
   * row does not represent something with a nav ord.
   */
  function navOrdForSubject(row) {
    var subject = row.getSubject(),
      ord;
    if (typeof subject.getNavOrd === 'function') {
      ord = subject.getNavOrd();
    }
    return ord ? ord.relativizeToSession() : null;
  }

  /**
   * Take a cell's existing html and wrap it in an anchor with an href derived
   * from a component's nav ord.
   */
  function addAnchorHtml(dom, navOrd) {
    var uri = navOrd.toUri(),
      html = dom.html(),
      $a = $('<a/>').addClass('link').attr('href', uri).attr('data-ord', navOrd).html(html);
    dom.html($a);
  }

  /**
   * A mixin to allow a column's content to be wrapped in an anchor
   * tag that will hyperlink to row's subject component.
   *
   * @alias module:webEditors/rc/wb/mixin/ComponentHyperlinkColumnMixin
   *
   * @mixin
   * @param {module:nmodule/webEditors/rc/wb/table/model/Column} target The
   * column instance that will have the mixin applied to it.
   * @param params
   * @param {String} params.cssClass an optional css class string that will be applied
   * to the cell if an anchor is successfully constructed for the row component's ord.
   */
  var exports = function exports(target, params) {
    var superBuildCell = target.buildCell,
      superDestroyCell = target.destroyCell,
      cssClass = params && params.cssClass;
    if (!(target instanceof Column)) {
      throw new Error('Mixin must be applied to a Column instance.');
    }
    if (!applyMixin(target, MIXIN_NAME)) {
      return;
    }

    /**
     * Extend the column's buildCell() functionality with the ability to wrap the generated
     * content in an anchor that, when clicked, will hyperlink to the component that is the
     * given `Row`'s subject.
     *
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @returns {Promise|*}
     */
    target.buildCell = function (row, dom) {
      var that = this;
      return Promise.resolve(superBuildCell.apply(that, arguments)).then(function () {
        var navOrd = navOrdForSubject(row);
        if (navOrd) {
          addAnchorHtml(dom, navOrd);
          if (cssClass) {
            dom.addClass(cssClass);
          }
          dom.on('click.hyperlink', 'a', function (e) {
            if (e.button !== 3) {
              var ord = $(this).data('ord');
              if (ord && typeof niagara !== 'undefined') {
                niagara.env.hyperlink(ord);
                return false;
              }
            }
            return true;
          });
        }
      });
    };

    /**
     * Extend the column's destroyCell() functionality with code to clean
     * up the click event handler added in the buildCell() function.
     *
     * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
     * @param {JQuery} dom
     * @returns {Promise|*}
     */
    target.destroyCell = function (row, dom) {
      var $a;

      // To destroy this cell, we first remove the click handler from the anchor,
      // then remove the anchor and replace the html, so the super destroyCell()
      // call will see the dom as it was before we wrapped the <td>'s content in
      // the anchor.

      $a = dom.find('a.link').first();
      if ($a.data('ord')) {
        dom.off('click.hyperlink');
        dom.html($a.html());
      }
      return Promise.resolve(superDestroyCell.call(this, row, dom));
    };
  };
  return exports;
});
