/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * @module nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn
 */
define(['nmodule/webEditors/rc/fe/baja/util/facetsUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/wb/mgr/model/MgrColumn', 'nmodule/webEditors/rc/wb/table/model/columns/PropertyColumn'], function (facetsUtils, typeUtils, compUtils, MgrColumn, PropertyColumn) {
  'use strict';

  var toProperties = facetsUtils.toProperties,
    getSlotFacets = typeUtils.getSlotFacets;

  /**
   * API Status: **Development**
   *
   * `MgrColumn` subclass that allows individual property values to be edited
   * on a component.
   *
   * The column name will correspond directly to the slot name to modify.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/mgr/model/columns/PropertyMgrColumn
   * @extends module:nmodule/webEditors/rc/wb/table/model/columns/PropertyColumn
   * @mixes module:nmodule/webEditors/rc/wb/mgr/model/MgrColumn
   */
  var PropertyMgrColumn = function PropertyMgrColumn() {
    PropertyColumn.apply(this, arguments);
  };
  PropertyMgrColumn.prototype = Object.create(PropertyColumn.prototype);
  PropertyMgrColumn.prototype.constructor = PropertyMgrColumn;
  MgrColumn.mixin(PropertyMgrColumn);

  /**
   * If editing only one row, then the facets used to configure the field
   * editor will be taken from the component instance in that one row.
   *
   * If editing multiple rows, then the facets will be taken from the
   * appropriate frozen slot on the column's configured Type, if present.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/table/model/Row>} rows
   * @returns {Object} config object for `fe.makeFor`, possibly including
   * properties derived from slot facets
   */
  PropertyMgrColumn.prototype.getConfigFor = function (rows) {
    var value = this.coalesceRows(rows),
      type = this.$type,
      name = this.getName(),
      facets;

    //c.f. javax.baja.workbench.mgr.MgrColumn.Prop#toEditor
    if (rows.length === 1) {
      facets = rows[0].getSubject().getFacets(name);
    } else {
      try {
        facets = type && getSlotFacets(type, name);
      } catch (ignore) {
        //dynamic slot so could not retrieve slot facets
      }
    }
    return {
      value: value,
      properties: toProperties(facets)
    };
  };

  /**
   * Sets/adds the property value on the row's `Component` subject.
   *
   * @param {baja.Value} value
   * @param {module:nmodule/webEditors/rc/wb/table/model/Row} row
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise} promise to be resolved when the value has
   * been set on the backing `Component`
   */
  PropertyMgrColumn.prototype.commit = function (value, row, params) {
    var progressCallback = params && params.progressCallback,
      prom = compUtils.writeSlot(row.getSubject(), this.getName(), value, params);
    if (progressCallback) {
      progressCallback(MgrColumn.COMMIT_READY);
    }
    return prom;
  };
  return PropertyMgrColumn;
});
