/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/*eslint-env browser */ /*jshint browser: true */

define(['baja!', 'jquery', 'underscore', 'lex!baja', 'css!nmodule/webEditors/rc/fe/webEditors-structure'], function (baja, $, _, lexs) {
  "use strict";

  /**
   * API Status: **Private**
   *
   * Utility functions for working with BStatus instances in manager views.
   *
   * @exports module:nmodule/webEditors/rc/wb/mgr/componentStatusUtils
   */
  var exports = {};

  ////////////////////////////////////////////////////////////////
  // Support
  ////////////////////////////////////////////////////////////////

  /**
   * Assemble an object containing the raw color data from the baja lexicon.
   */
  var getObj = _.once(function () {
    var obj = {},
      bajaLex = lexs[0],
      statuses = ['disabled', 'fault', 'down', 'alarm', 'stale', 'overridden'];
    _.each(statuses, function (status) {
      obj[status] = {
        fg: bajaLex.get('Status.' + status + '.fg'),
        bg: bajaLex.get('Status.' + status + '.bg')
      };
    });
    return obj;
  });

  /**
   * Function to override the default color values in the css file,
   * allowing the user to configure the status styles from the lexicon
   * as they can in Workbench, without the need to create a new theme.
   * This function is intended to be called lazily, at the point the css
   * is first applied to a dom element.
   */
  var configureCss = _.once(function () {
    var statusStyle = document.createElement('style');
    $.each(getObj(), function (status, colors) {
      statusStyle.innerHTML += '.component-status-' + status + ' { color: ' + colors.fg + '; background-color: ' + colors.bg + '}\n';
    });
    document.head.appendChild(statusStyle);
  });
  exports.$toSimpleString = function (status) {
    // Check the status and add a style for anything other than 'ok'. The ordering
    // here reflects the same ordering of the getBackgroundColor() method in the
    // BStatus class in the bajaui code.

    if (baja.hasType(status, 'baja:Status')) {
      if (status.isDisabled()) {
        return 'disabled';
      } else if (status.isFault()) {
        return 'fault';
      } else if (status.isDown()) {
        return 'down';
      } else if (status.isAlarm()) {
        return 'alarm';
      } else if (status.isStale()) {
        return 'stale';
      } else if (status.isOverridden()) {
        return 'overridden';
      }
    }
  };

  /**
   * Get the foreground color configured in the baja lexicon for this status.
   * @param {baja.Status} status
   * @returns {string|undefined}
   */
  exports.getForeground = function (status) {
    var str = exports.$toSimpleString(status);
    return str && getObj()[str].fg;
  };

  /**
   * Get the background color configured in the baja lexicon for this status.
   * @param {baja.Status} status
   * @returns {string|undefined}
   */
  exports.getBackground = function (status) {
    var str = exports.$toSimpleString(status);
    return str && getObj()[str].bg;
  };

  /**
   * Get a CSS class appropriate to be applied to a DOM element for this status.
   * @param {baja.Status} status
   * @returns {string|undefined}
   */
  exports.getStatusCssClass = function (status) {
    configureCss();
    var str = exports.$toSimpleString(status);
    return str && 'component-status-' + str;
  };

  /**
   * Function to apply a css style to a dom element according to a subject's
   * current status.
   *
   * @param {baja.Status} status
   * @param {JQuery} dom A jQuery object for the subject's DOM.
   */
  exports.addComponentStatusCss = function (status, dom) {
    var cssClass = exports.getStatusCssClass(status);
    if (cssClass) {
      dom.addClass(cssClass);
    }
  };
  return exports;
});
