/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 */

/*jshint browser:true*/

/**
 * A mixin type, used to add learn functionality to a Manager instance.
 *
 * @module nmodule/webEditors/rc/wb/mgr/MgrLearn
 */
define(['baja!', 'log!nmodule.webEditors.rc.wb.mgr.MgrLearn', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/DepthSubscriber', 'nmodule/webEditors/rc/wb/mixin/mixinUtils', 'nmodule/webEditors/rc/wb/mgr/Manager', 'nmodule/webEditors/rc/wb/mgr/MgrLearnTableSupport', 'nmodule/webEditors/rc/wb/mgr/mgrUtils', 'nmodule/webEditors/rc/wb/mgr/commands/AddCommand', 'nmodule/webEditors/rc/wb/mgr/commands/CancelDiscoverCommand', 'nmodule/webEditors/rc/wb/mgr/commands/DiscoverCommand', 'nmodule/webEditors/rc/wb/mgr/commands/LearnModeCommand', 'nmodule/webEditors/rc/wb/mgr/commands/MatchCommand', 'nmodule/webEditors/rc/wb/table/tree/TreeTable'], function (baja, log, Promise, _, DepthSubscriber, mixinUtils, Manager, mixinLearnTableSupport, mgrUtils, AddCommand, CancelDiscoverCommand, DiscoverCommand, LearnModeCommand, MatchCommand, TreeTable) {
  'use strict';

  var applyMixin = mixinUtils.applyMixin,
    findCommand = mgrUtils.findCommand,
    logError = log.severe.bind(log),
    MIXIN_NAME = 'MGR_LEARN';

  /**
   * API Status: **Development**
   *
   * A mixin to provide learn support to a bajaux manager view.
   *
   * To support discovery, in addition to applying this mixin, the target manager object must
   * provide several functions that this mixin will use to accomplish the discovery and the
   * creation of new components from the discovered items.
   *
   * The concrete manager must provide a `makeLearnModel()` method. This should return a
   * `Promise` that will resolve to a `TreeTableModel`. This will be used as the data model
   * for the discovery table. On completion of the discovery job, the manager should use the
   * result of the job to insert items into the discovery model.
   *
   * The concrete manager must also provide an implementation of a `doDiscover()` function
   * that will create a job (typically by invoking an action that will submit a job
   * and return the ord), and then set the job on the manager via the `setJob()` function.
   * This function will accept the job instance or the ord for a job, specified either as
   * a `baja.Ord` or a string.
   *
   * Once the job is complete, a 'jobcomplete' tinyevent will be emitted on the manager. The
   * concrete manager will also typically have a handler for that event, which will get the
   * discovered items from the job by some means, and then update the discovery table. This
   * will normally involve inserting nodes into the learn model. The manager may store arbitrary
   * data on those nodes, which it may retrieve later via the node's `value()` function.
   *
   * The manager must also implement a `getTypesForDiscoverySubject()` function. This will be called
   * when dragging an item from the discovery table to the database table or invoking the 'add'
   * command. The function may be called several times, each time its argument will be a
   * `TreeNode` representing the item to be added into to the database table. The implementation
   * of this function is expected to return a single `MgrTypeInfo` instance or any array of them.
   * These will be used to create a new component instance of the required type for the discovered
   * node.
   *
   * Also to support the addition of new components, the manager should implement a function
   * called `getProposedValuesFromDiscovery()`. This will be passed the tree node that was dragged
   * from the discovery table to the database table. The function should obtain any information
   * the manager had set on the node at discovery time and use it to create an object containing
   * the initial values for the new component. The names of properties on the object returned by
   * the function will be compared against the column names in the main database model. For the
   * columns that have matching names, the values of those properties will be used to set the
   * initial proposed values on the new row(s) when the dialog for editing the new instances is
   * displayed.
   *
   * @alias module:nmodule/webEditors/rc/wb/mgr/MgrLearn
   * @mixin
   * @extends module:nmodule/webEditors/rc/wb/mgr/Manager
   * @param {module:nmodule/webEditors/rc/wb/mgr/Manager} target
   *
   * @example
   * <caption>Add the MgrLearn mixin to a Manager subclass to add learn
   * functionality.</caption>
   * require([...'nmodule/webEditors/rc/wb/mgr/MgrLearn'], function (...MgrLearn) {
   *   function MyManager() {
   *     Manager.apply(this, arguments);
   *     MgrLearn(this);
   *   }
   *   MyManager.prototype = Object.create(Manager.prototype);
   *
   *   //implement abstract functions
   *   MyManager.prototype.doDiscover = function () { ...
   * });
   */
  var MgrLearn = function MgrLearn(target, params) {
    if (!(target instanceof Manager)) {
      throw new Error('target must be a Manager instance.');
    }
    if (!applyMixin(target, MIXIN_NAME, MgrLearn.prototype)) {
      return this;
    }
    params = _.defaults(params || {}, {
      tableCtor: TreeTable
    });
    var superDoDestroy = target.doDestroy;

    /**
     * Extension of the manager's doDestroy() function. This will clean up
     * the subscription to the discovery job, if it is present.
     *
     * @returns {*|Promise}
     */
    target.doDestroy = function () {
      var that = this;
      if (that.$jobSubDepth) {
        delete that.$jobSubDepth;
      }
      return unsubscribeJob(that).then(function () {
        return superDoDestroy.apply(that, arguments);
      });
    };

    // Automatically mix in support for the learn table, so the target
    // manager does not need to do that themselves (or know about that
    // module...)

    mixinLearnTableSupport(target, {
      tableCtor: params.tableCtor
    });
  };

  /**
   * Abstract method used to obtain the model for the learn tree table. This
   * should return a `TreeTableModel`, or a Promise that resolves to one.
   *
   * @abstract
   * @returns {Promise.<module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel>} a
   * tree table model that will be used by the manager's discovery table.
   */
  MgrLearn.prototype.makeLearnModel = function () {
    throw new Error('makeLearnModel() function not implemented.');
  };

  /**
   * Abstract method used to initiate the discovery process. What this
   * implementation does is a matter for the concrete manager, but the typical
   * pattern will be to invoke an Action that will submit a job, and then set
   * that job or its Ord on the manager via the `#setJob()` function.
   *
   * @abstract
   * @returns {Promise|*} Optionally return a Promise
   */
  MgrLearn.prototype.doDiscover = function () {
    throw new Error('doDiscover() function not implemented.');
  };

  /**
   * Abstract method to get the Component type(s) that could be created for the
   * given discovery node when adding it to the station as a component. If
   * returning an array, the first element of the array should be the type that
   * represents the best mapping for the discovery item.
   *
   * @abstract
   * @param {*} discovery a discovery object
   * @returns {Promise.<Array.<string>>} an array of TypeSpecs that could be
   * constructed from this node, or a Promise resolving to one
   */
  MgrLearn.prototype.getTypesForDiscoverySubject = function (discovery) {
    throw new Error('getTypesForDiscoverySubject() function not implemented.');
  };

  /**
   * Abstract method to get the initial values for a discovered node when it is
   * being added to the station as a new component. This method should return an
   * Object instance, with the values to be used by the new instances. The
   * returned object may have a property called 'name', which will be used to
   * set the slot name of the new component. It may also have a child object
   * named 'values'. Each property of this object with a name that matches the
   * name of a `Column` in the main table model will have that property's value
   * used as the initial value when the component editor is displayed.
   *
   * @example
   * <caption>Return the initial values for the component name, and the
   * 'version' and 'address' columns</caption>
   * MyDeviceMgr.prototype.getProposedValuesFromDiscovery = function (discovery) {
     *   return {
     *     name: discovery.deviceName,
     *     values: {
     *       address: discovery.address,
     *       version: discovery.firmwareVersionMajor + '.' + discovery.firmwareVersionMinor
     *     }
     *   };
     * };
   *
   * @abstract
   * @param {*} discovery an object obtained from a node in discovery table.
   * @param {*} subject - the subject of the `Row` whose values are to be
   * proposed.
   * @see module:nmodule/webEditors/rc/wb/table/tree/TreeNode
   * @returns {Object|Promise.<Object>} an object literal with the name and
   * initial values to be used for the new component.
   */
  MgrLearn.prototype.getProposedValuesFromDiscovery = function (discovery, subject) {
    throw new Error('getProposedValuesFromDiscovery() function not implemented.');
  };

  /**
   * @function module:nmodule/webEditors/rc/wb/mgr/MgrLearn#isExisting
   * @param {*} discovery the discovery item
   * @param {baja.Component} component component already existing in local
   * database
   * @returns {boolean|Promise.<boolean>} true if the local component already
   * represents the discovery item
   */

  /**
   * Get the learn model. The model will have been created via a call to `makeLearnModel()`; a
   * function that the concrete manager must provide. This will return the `TreeTableModel`
   * resolved from the Promise.
   *
   * @returns {module:nmodule/webEditors/rc/wb/table/tree/TreeTableModel}
   */
  MgrLearn.prototype.getLearnModel = function () {
    return this.$learnModel;
  };

  /**
   * Begin the discovery process. This function is not intended to be called directly
   * or overridden by concrete manager types. Instead, the concrete manager should provide
   * a `doDiscover` function, which will be called from this function. If the function is
   * not provided, an Error will be thrown.
   *
   * @private
   */
  MgrLearn.prototype.discover = function () {
    var that = this,
      modeCmd = that.$getLearnModeCommand(),
      prom;

    // Set the learn mode command to selected, if it isn't already. This will cause
    // the discovery pane to become visible, and the table heights to be adjusted.

    if (modeCmd && !modeCmd.isSelected()) {
      prom = Promise.resolve(modeCmd.invoke());
    } else {
      prom = Promise.resolve();
    }
    return prom.then(function () {
      return that.doDiscover();
    });
  };

  /**
   * Attach a job to this manager, typically as part of a driver discovery process.
   * The act of attaching a job will subscribe to it, and cause a 'jobcomplete' event
   * to be emitted once the job is complete. A manager will typically update the learn
   * model at that point.
   *
   * @param {Object} params an Object literal containing the parameters for this function.
   *
   * @param {string|baja.Ord|baja.Component} [params.jobOrOrd] either a BJob instance, an
   * Ord referencing a job, or a String containing the ord for a job.
   *
   * @param {Number} [params.depth] optional parameter that will be used when subscribing to
   * the job once it has completed; this allows the job plus its final set of children to
   * be subscribed. A depth of 1 will subscribe to the job, 2 will subscribe the job and
   * its children, and so on. Subscription will default to a depth of 1 if this parameter
   * is not specified.
   *
   * @returns {Promise}
   */
  MgrLearn.prototype.setJob = function (params) {
    var that = this,
      jobOrOrd,
      depth,
      isJob;
    params = baja.objectify(params, 'jobOrOrd');
    jobOrOrd = params.jobOrOrd;
    depth = params.depth || 1;

    // First, clean up any existing job that might already have been set on the manager.
    // After that, obtain the new job: we might have been passed one directly, or we might
    // need to resolve an ord.

    return unsubscribeJob(that).then(function () {
      return baja.station.sync();
    }).then(function () {
      if (that.$learnJob) {
        delete that.$learnJob;
      }
      isJob = baja.hasType(jobOrOrd, 'baja:Job');
      return isJob ? Promise.resolve(jobOrOrd) : baja.Ord.make(String(jobOrOrd)).get();
    }).then(function (job) {
      var discoveryCmd = that.$getDiscoveryCommand(),
        cancelCmd = that.$getCancelDiscoveryCommand();

      // Now we have a reference to the job, keep track of it with some private
      // properties, then load it into the job bar.

      that.$learnJob = job;
      that.$jobSubDepth = depth;
      if (discoveryCmd) {
        discoveryCmd.setEnabled(false);
      }
      if (cancelCmd) {
        cancelCmd.setEnabled(true);
      }
      return that.$getJobBar().load(job).then(function () {
        return job;
      });
    }).then(function (job) {
      // Test if the job has already completed, if so, call the
      // job complete callback directly. Otherwise, subscribe and
      // wait for an event to notify us that the job has finished.

      if (isJobComplete(job)) {
        // The job has already completed, so subscribe to the job
        // and its children to the requested depth. We don't need
        // to pass in a change callback, as once it's complete we
        // don't care about changes.

        return subscribeJob(that, job, depth).then(function () {
          jobComplete(that, job);
        });
      } else {
        // The job is not yet complete, so subscribe to it with a
        // depth of 1. Once the job is complete, we'll resubscribe
        // to it and its children to the requested depth. We pass
        // a callback here, so we can get the change events.

        return subscribeJob(that, job, 1, function (p) {
          jobStateChanged(that, p);
        });
      }
    });
  };

  /**
   * Get the discovery job currently set against the manager.
   *
   * @returns {baja.Component}
   */
  MgrLearn.prototype.getJob = function () {
    return this.$learnJob;
  };

  /**
   * Creates a new component instance from the types the manager specified
   * for a particular node in the discovery table. If the manager returned
   * more than one type, this default implementation will return a new
   * instance based on the first type info.
   *
   * @param {*} discovery an instance of a discovery object (e.g. an
   * `ndriver:NDiscoveryLeaf`), dragged from the discovery table and dropped
   * onto the database table or selected when the 'Add' command was invoked.
   *
   * @param {Array.<module:nmodule/webEditors/rc/wb/mgr/MgrTypeInfo>} typeInfos - an
   * array of MgrTypeInfos, created from the type or types returned
   * by the manager's `getTypesForDiscoverySubject()` implementation.
   *
   * @returns {Promise} a Promise of new component instance for the discovered item
   * based on the provided type information.
   */
  MgrLearn.prototype.newInstanceFromDiscoverySubject = function (discovery, typeInfos) {
    return this.getModel().newInstance(_.head(typeInfos));
  };

  /**
   * Search for the existing component that matches the given node from the
   * discovery table. To match a component, the concrete manager subclass
   * must contain a function named `isExisting()` which will be passed the
   * discovery object and a component. The function will be used as a predicate and
   * should return true if the given component represents the same item as
   * the discovery table item, false otherwise. If the manager does not provide
   * such a function, all discovery nodes will be considered as not matching any
   * existing components.
   *
   * @param {*} discovery a discovered object
   *
   * @returns {Promise.<baja.Component|undefined>} the existing component that was found to match
   * the given discovery node, or undefined if no such match was found.
   */
  MgrLearn.prototype.getExisting = function (discovery) {
    var that = this,
      comps = _.invoke(that.getModel().getRows(), 'getSubject');
    if (typeof that.isExisting === 'function') {
      return Promise.resolve(_.find(comps, function (c) {
        return !!that.isExisting(discovery, c);
      }));
    }
    return Promise.resolve(undefined);
  };

  ////////////////////////////////////////////////////////////////
  // Discovery Commands
  ////////////////////////////////////////////////////////////////

  /**
   * Creates and returns an array of discovery related commands.
   * These are the LearnModeCommand (show/hide the learn pane),
   * DiscoverCommand, CancelDiscoverCommand, AddCommand, and MatchCommand.
   *
   * @returns {Array.<module:bajaux/commands/Command>} a new array containing
   * the discovery related commands.
   */
  MgrLearn.prototype.makeDiscoveryCommands = function () {
    var that = this;
    return [new LearnModeCommand(that), new DiscoverCommand(that), new CancelDiscoverCommand(that), new AddCommand(that), new MatchCommand(that)];
  };

  /**
   * Get the 'LearnModeCommand' instance from the command group. This will be used by the
   * discovery command to enable the learn pane if a discovery is started.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/LearnModeCommand}
   */
  MgrLearn.prototype.$getLearnModeCommand = function () {
    return findCommand(this, LearnModeCommand);
  };

  /**
   * Get the 'DiscoverCommand' instance from the command group. Invoking this will show
   * the learn pane and call the `discover()` function on the manager.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/DiscoverCommand}
   */
  MgrLearn.prototype.$getDiscoveryCommand = function () {
    return findCommand(this, DiscoverCommand);
  };

  /**
   * Return the 'CancelDiscoveryCommand' instance from the command group. This is used
   * to enable or disable the command as the discovery process starts or stops.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/CancelDiscoverCommand}
   */
  MgrLearn.prototype.$getCancelDiscoveryCommand = function () {
    return findCommand(this, CancelDiscoverCommand);
  };

  /**
   * Return the 'AddCommand' instance from the command group. This is used
   * to add selected items from the discovery table into the database table.
   * It's also invoked from the drag and drop operation.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/AddCommand}
   */
  MgrLearn.prototype.$getAddCommand = function () {
    return findCommand(this, AddCommand);
  };

  /**
   * Return the 'MatchCommand' instance from the command group. This will
   * can be used to update an existing database item from a discovered item.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/mgr/commands/MatchCommand}
   */
  MgrLearn.prototype.$getMatchCommand = function () {
    return findCommand(this, MatchCommand);
  };

  /**
   * Test for discovery job completion.
   * @private
   */
  function isJobComplete(job) {
    return !baja.AbsTime.DEFAULT.equals(job.get('endTime'));
  }

  /**
   * Callback function used for receiving notifications of changes on the discovery
   * job. This is used to listen for notification of the job completion, at which
   * point we will emit the 'jobcomplete' event.
   *
   * @private
   * @param mgr the manager instance
   * @param prop the property that has changed on the subscribed job.
   */
  function jobStateChanged(mgr, prop) {
    var job = mgr.$learnJob;

    // If the end time has changed, we'll look to see whether the job is
    // now finished. If so, we'll emit the event on the manager. We'll
    // also re-subscribe to the requested depth provided in the setJob()
    // call. We'll previously have been subscribed at depth 1, but we can
    // now subscribe to the job's children.

    if (prop.getName() === 'endTime' && isJobComplete(job)) {
      unsubscribeJob(mgr).then(function () {
        return subscribeJob(mgr, job, mgr.$jobSubDepth);
      }).then(function () {
        jobComplete(mgr, job);
      })["catch"](logError);
    }
  }

  /**
   * Subscribe to the discovery job that has been attached to the manager.
   * If the job has not yet completed, the depth parameter is expected to
   * be 1, to listen for changes to the job's properties. If the job is complete,
   * the depth will be the value specified during the call to setJob(), as at
   * that point we are in a position to be able to subscribe to the job's children,
   * if needed. There is an optional change callback parameter - this is used to get
   * our notification that the job has finished.
   *
   * @private
   * @param mgr
   * @param {baja.Component} job
   * @param {Number} depth
   * @param {Function} [changeCallback]
   * @returns {Promise}
   */
  function subscribeJob(mgr, job, depth, changeCallback) {
    var subscriber = new DepthSubscriber(depth);
    if (changeCallback) {
      subscriber.attach('changed', changeCallback);
    }
    mgr.$jobSubscriber = subscriber;
    return subscriber.subscribe(job);
  }

  /**
   * Unsubscribe from the job and remove the property on the manager.
   *
   * @private
   * @param mgr
   * @returns {Promise}
   */
  function unsubscribeJob(mgr) {
    var subscriber = mgr.$jobSubscriber;
    if (subscriber) {
      subscriber.detach();
      delete mgr.$jobSubscriber;
      return subscriber.unsubscribeAll();
    }
    return Promise.resolve();
  }

  /**
   * Function called when a job attached via `setJob()` has completed. This
   * will emit a 'jobcomplete' event to any registered handlers.
   *
   * @private
   * @param {module:nmodule/webEditors/rc/wb/mgr/MgrLearn} mgr
   * @param {baja.Component} job the completed job.
   */
  function jobComplete(mgr, job) {
    var discoveryCmd = mgr.$getDiscoveryCommand(),
      cancelCmd = mgr.$getCancelDiscoveryCommand();
    if (discoveryCmd) {
      discoveryCmd.setEnabled(true);
    }
    if (cancelCmd) {
      cancelCmd.setEnabled(false);
    }
    mgr.emit('jobcomplete', job);
  }
  return MgrLearn;
});
