function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*global niagara */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/links/OrdChooser
 */
define(['baja!', 'lex!webEditors', 'log!nmodule.webEditors.rc.wb.links.OrdChooser', 'bajaux/commands/Command', 'bajaux/util/CommandButtonGroup', 'bajaux/Widget', 'Promise', 'underscore', 'nmodule/js/rc/switchboard/switchboard', 'nmodule/webEditors/rc/fe/fe', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/IconEditor', 'nmodule/webEditors/rc/fe/baja/util/spaceUtils', 'nmodule/webEditors/rc/util/htmlUtils', 'nmodule/webEditors/rc/util/textUtils', 'nmodule/webEditors/rc/wb/tree/BajaComplexTreeNode', 'nmodule/webEditors/rc/wb/tree/NavTree', 'nmodule/webEditors/rc/wb/tree/stationTree', 'hbs!nmodule/webEditors/rc/wb/links/template/OrdChooser'], function (baja, lexs, log, Command, CommandButtonGroup, Widget, Promise, _, switchboard, fe, BaseEditor, IconEditor, spaceUtils, htmlUtils, textUtils, BajaComplexTreeNode, NavTree, stationTree, tplOrdChooser) {
  'use strict';

  var VALUE_READY_EVENT = BaseEditor.VALUE_READY_EVENT,
    SESSION_NODE_NAME = stationTree.SESSION_NODE_NAME,
    DEFAULT_ORD = baja.Ord.DEFAULT,
    makeOrd = baja.Ord.make,
    webEditorsLex = lexs[0],
    logError = log.severe.bind(log),
    resolveStation = spaceUtils.resolveStation,
    selectOnFocus = htmlUtils.selectOnFocus,
    selectAll = htmlUtils.selectAll,
    ROOT_NODE_PROPERTY = 'rootNode',
    SELECT_FILTER_PROPERTY = 'selectFilter',
    DISPLAY_FILTER_PROPERTY = 'displayFilter',
    subjectToDisplay = textUtils.subjectToDisplay;

  ////////////////////////////////////////////////////////////////
  // Utility functions
  ////////////////////////////////////////////////////////////////

  /**
   * Get the display name for the currently selected ORD.
   *
   * @param {Object} ordTarget ORD target (from `baja.Ord#resolve`)
   * @returns {Promise}
   */
  function getDisplayName(ordTarget) {
    var object = ordTarget.object,
      container = ordTarget.container,
      propertyPath = ordTarget.propertyPath || [],
      slot = ordTarget.slot,
      displayName,
      i;
    if (object && typeof object.toConfig === 'function') {
      return subjectToDisplay(object);
    } else {
      if (baja.hasType(object, 'baja:Station')) {
        displayName = webEditorsLex.get('stationTree.config.displayName');
      } else if (baja.hasType(object, 'hierarchy:HierarchySpace')) {
        displayName = webEditorsLex.get('stationTree.hierarchy.displayName');
      } else if (baja.hasType(object, 'history:HistorySpace')) {
        displayName = webEditorsLex.get('stationTree.histories.displayName');
      } else if (baja.hasType(object, 'baja:IFile')) {
        displayName = object.getFileName();
      } else if (baja.hasType(object, 'baja:INavNode')) {
        displayName = object.getNavDisplayName();
      } else if (baja.hasType(object, 'baja:Complex')) {
        displayName = object.getDisplayName();
      } else if (slot && slot.isProperty()) {
        for (i = 0; i < propertyPath.length - 1; i++) {
          container = container.get(propertyPath[i]);
        }
        displayName = container.getDisplayName(propertyPath[i]);
      } else if (slot) {
        //action or topic
        displayName = container.getDisplayName(slot);
      }
      return Promise.resolve(displayName);
    }
  }
  function getIcon(ordTarget) {
    var object = ordTarget.object,
      slot = ordTarget.slot;
    //TODO: baja.File#getNavIcon
    if (baja.hasType(object, 'baja:INavNode') && object.getNavIcon) {
      return object.getNavIcon();
    }
    return object || slot;
  }

  /**
   * Get the ORD described by an ORD target.
   *
   * @param {Object} ordTarget ORD target (from `baja.Ord#resolve`)
   * @returns {baja.Ord}
   */
  function getTargetOrd(ordTarget) {
    var comp = ordTarget.getComponent() || ordTarget.getObject(),
      ord = String(comp && comp.getNavOrd());
    if (ordTarget.propertyPath) {
      if (ord.charAt(ord.length - 1) !== '/') {
        ord += '/';
      }
      ord += ordTarget.propertyPath.join('/');
    } else if (ordTarget.slot) {
      if (ord.charAt(ord.length - 1) !== '/') {
        ord += '/';
      }
      ord += ordTarget.slot;
    }
    return makeOrd(ord);
  }

  /**
   * @param {baja.Ord} ord
   * @returns {boolean}
   */
  function isHierarchyOrd(ord) {
    try {
      return !!ord.parse().get('hierarchy');
    } catch (e) {
      return false;
    }
  }

  /**
   * @param {baja.Ord} ord
   * @returns {boolean}
   */
  function isHistoryOrd(ord) {
    try {
      return !!ord.parse().get('history');
    } catch (e) {
      return false;
    }
  }

  /**
   * Get the raw ORD display string.
   *
   * @param {Object} ordTarget ORD target (from `baja.Ord#resolve`)
   * @param {baja.Ord} origOrd the ORD from which the target was resolved
   * @returns {string}
   */
  function getOrdDisplay(ordTarget, origOrd) {
    var ord;
    if (isHierarchyOrd(origOrd) || isHistoryOrd(origOrd)) {
      //for a history or hierarchy ord, always use the original, not an ord derived from
      //the resolved component in the station
      ord = origOrd;
    } else {
      ord = getTargetOrd(ordTarget);
    }
    return String(relativize(ord));
  }

  //TODO: this needs to be shared between this and profile.js, and probably
  // other stuff too
  /**
   * Given an ORD, convert it to a path of node names. This will be used to
   * preselect a node in the component tree.
   *
   * @param {baja.Ord} ord
   * @param {baja.Component} ordBase
   * @returns {Promise.<Array.<String>>}
   */
  function ordToPath(ord, ordBase) {
    var body = SESSION_NODE_NAME;
    if (ord.equals(DEFAULT_ORD)) {
      return Promise.resolve([body]);
    }
    var query = relativize(ord).parse(),
      slotPath = query.get('slot'),
      file = query.get('file'),
      hierarchy = query.get('hierarchy'),
      history = query.get('history');
    if (slotPath) {
      return resolveStation(ordBase).then(function (station) {
        return [body, String(relativize(station.getNavOrd()))].concat(slotPath.getBody().substring(1).split('/'));
      });
    } else if (hierarchy) {
      body += '/hierarchy' + hierarchy.getBody(); //body starts with '/'
      return Promise.resolve(body.split('/').map(baja.SlotPath.unescape));
    } else if (file) {
      //strip leading "^" - it's assumed we're looking in the file root
      body += '/file/' + file.getBody().replace(/^\^/, '');
    } else if (history) {
      body += '/history' + history.getBody(); //body starts with '/'
      return Promise.resolve(body.split('/').map(baja.SlotPath.unescape));
    } else {
      body += '/' + ord;
    }
    var path = body.split('/');
    if (path[path.length - 1] === "") {
      path.pop();
    }
    return Promise.resolve(path);
  }

  /**
   * Given a tree node (gotten from the component tree when a node is selected
   * by the user), convert it to the corresponding ORD.
   *
   * @param {module:nmodule/webEditors/rc/wb/tree/BajaNavTreeNode} node
   * @param {baja.Ord} [baseOrd]
   * @returns {baja.Ord|undefined} ORD derived from the selected node's path,
   * or undefined if no actual ORD is selected
   */
  function nodeToOrd(node, baseOrd) {
    var path = node.getFullPath();
    var ord = path[1]; //start from index 1 - skip root node from stationTree

    switch (ord) {
      case 'file':
        ord = 'file:^';
        break;
      case 'hierarchy':
        //TODO: ask Eric - is this right?
        return relativize(node.value().getNavOrd());
      case 'history':
        return relativize(node.value().getNavOrd());
    }
    if (!ord) {
      return DEFAULT_ORD;
    } else {
      if (baseOrd) {
        ord = baseOrd.toString() + "/";
      }
      var fullPath = ord + path.slice(2).join('/');
      if (fullPath.charAt(fullPath.length - 1) === '/') {
        fullPath = fullPath.substr(0, fullPath.length - 1);
      }
      return relativize(fullPath);
    }
  }

  /**
   * @param {baja.Ord|string} ord
   * @returns {baja.Ord}
   */
  function relativize(ord) {
    return makeOrd(ord).relativizeToSession();
  }

  /**
   * Return true if the given ORD can be resolved against the station.
   *
   * @param {baja.Ord} ord
   * @returns {boolean}
   */
  function shouldResolve(ord) {
    if (!ord) {
      return false;
    }
    try {
      var list = ord.parse();
      return !!(list.get('slot') || list.get('h') || list.get('file') || list.get('hierarchy') || list.get('history'));
    } catch (e) {
      return false;
    }
  }

  ////////////////////////////////////////////////////////////////
  // Commands
  ////////////////////////////////////////////////////////////////

  var ChangeCommand = function ChangeCommand(ordChooser) {
    Command.call(this, {
      module: 'webEditors',
      lex: 'OrdChooser.changeCommand',
      func: function func() {
        return ordChooser.$toggleNavTree();
      }
    });
  };
  ChangeCommand.prototype = Object.create(Command.prototype);
  ChangeCommand.prototype.constructor = ChangeCommand;
  var HyperlinkCommand = function HyperlinkCommand(ordChooser) {
    Command.call(this, {
      module: 'webEditors',
      lex: 'OrdChooser.hyperlinkCommand',
      func: function func() {
        return ordChooser.read().then(function (ord) {
          return niagara.env.hyperlink(String(ord));
        });
      }
    });
  };
  HyperlinkCommand.prototype = Object.create(Command.prototype);
  HyperlinkCommand.prototype.constructor = HyperlinkCommand;

  ////////////////////////////////////////////////////////////////
  // OrdChooser definition
  ////////////////////////////////////////////////////////////////

  /**
   * Editor to allow the user to choose an ORD using a component tree retrieved
   * from the station.
   *
   * It supports the following bajaux properties:
   *
   * - `showOrdText`: set to `false` to cause the raw text of the ORD to be
   *   hidden. The user will not be able to type an ORD but must select one
   *   from the tree. Only the name of the tree node will be shown. Note that
   *   this property only applies with form factor `compact`.
   * - `rootNode`:  An ord to target a specific part of the station, files,
   *    or hierarchy. Example: file:^folder would make folder the top level node
   *    for OrdChooser.
   * - `selectFilter`: A function that returns a promise that resolves if an
   *    element meets selection criteria, and it rejects if not.
   * - `displayFilter`: A function that returns true if an element should be
   *    displayed, false if not. Parameters are as follows:
   *    function({module:nmodule/webEditors/rc/wb/tree/TreeNode} : parent,
   *            {module:nmodule/webEditors/rc/wb/tree/TreeNode} : child)
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/wb/links/OrdChooser
   *
   */
  var OrdChooser = function OrdChooser() {
    BaseEditor.apply(this, arguments);
    switchboard(this, {
      '$initializeNavTree': {
        allow: 'once',
        onRepeat: 'returnLast'
      },
      '$loadOrdIntoNavTree': {
        allow: 'oneAtATime',
        onRepeat: 'queue'
      },
      '$updateDisplay': {
        allow: 'oneAtATime',
        onRepeat: 'preempt'
      }
    });
    this.getCommandGroup().add(new ChangeCommand(this), new HyperlinkCommand(this));
    this.validators().add(function (ord) {
      if (ord.equals(DEFAULT_ORD)) {
        return;
      }
      ord.parse(); //ORD does not need to exist, but should be parseable
    });
  };
  OrdChooser.prototype = Object.create(BaseEditor.prototype);
  OrdChooser.prototype.constructor = OrdChooser;

  ////////////////////////////////////////////////////////////////
  // Private methods
  ////////////////////////////////////////////////////////////////

  OrdChooser.prototype.$getChangeCommand = function () {
    return this.getCommandGroup().get(0);
  };
  OrdChooser.prototype.$getHyperlinkCommand = function () {
    return this.getCommandGroup().get(1);
  };

  /**
   * Get the element containing the ORD display information.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getDisplayElement = function () {
    return this.jq().children('.display');
  };

  /**
   * Get the element containing the icon for the Type of the object the ORD
   * points to.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getIconElement = function () {
    return this.$getDisplayElement().children('.icon');
  };

  /**
   * Get the editor showing the Ord target's icon.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/IconEditor}
   */
  OrdChooser.prototype.$getIconEditor = function () {
    // noinspection JSValidateTypes
    return Widget["in"](this.$getIconElement());
  };

  /**
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getCommandGroupElement = function () {
    return this.jq().children('.edit').children('.buttons');
  };

  /**
   * @private
   * @returns {module:bajaux/util/CommandButtonGroup}
   */
  OrdChooser.prototype.$getCommandButtonGroup = function () {
    // noinspection JSValidateTypes
    return Widget["in"](this.$getCommandGroupElement());
  };

  /**
   * Get the element containing the display name for the object the ORD
   * points to.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getDisplayNameElement = function () {
    return this.$getDisplayElement().children('.displayName');
  };

  /**
   * Get the element containing display details about the ORD itself.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getDetailsElement = function () {
    return this.jq().children('.details');
  };

  /**
   * Get the element containing the raw text display of the ORD.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getOrdElement = function () {
    return this.$getDetailsElement().children('.ord');
  };

  /**
   * Get the element containing edit controls.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getEditElement = function () {
    return this.jq().children('.edit');
  };

  /**
   * Get the element containing the component tree.
   *
   * @private
   * @returns {JQuery}
   */
  OrdChooser.prototype.$getNavTreeElement = function () {
    return this.$getEditElement().children('.tree');
  };

  /**
   * Get the loaded NavTree instance for the component tree.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/wb/tree/NavTree|undefined} the
   * loaded NavTree instance, or `undefined` if the tree has not yet loaded
   */
  OrdChooser.prototype.$getNavTree = function () {
    // noinspection JSValidateTypes
    return Widget["in"](this.$getNavTreeElement());
  };
  OrdChooser.prototype.$selectAll = function () {
    selectAll(this.$getOrdElement());
  };

  /**
   * @private
   * @param {baja.Ord|String} ordDisplay the ORD text to display
   * @param {String} displayName the display name of the nav node the ORD points to
   */
  OrdChooser.prototype.$setText = function (ordDisplay, displayName) {
    this.$getOrdElement().val(String(ordDisplay));
    this.$getDisplayNameElement().text(displayName).attr('title', ordDisplay);
  };

  /**
   * Update the contents of all display elements to reflect the new ORD.
   *
   * @private
   * @param {baja.Ord} ord
   * @param {Array.<String>} [icon]
   * @returns {Promise} promise to be resolved after the ORD has been
   * resolved and queried for display information
   */
  OrdChooser.prototype.$updateDisplay = function (ord, icon) {
    var that = this,
      iconEditor = that.$getIconEditor();
    function setText(ordDisplay, displayName) {
      return that.$setText(ordDisplay, displayName);
    }
    if (that.getFormFactor() === Widget.formfactor.mini) {
      return setText(ord, '');
    }
    if (ord === DEFAULT_ORD) {
      return Promise.resolve(setText('null', 'null'));
    } else if (shouldResolve(ord)) {
      return ord.resolve({
        base: baja.station
      })["catch"](function (err) {
        setText(ord, '');
        throw err;
      }).then(function (target) {
        return getDisplayName(target).then(function (displayName) {
          setText(getOrdDisplay(target, ord), displayName);
          return iconEditor.load(icon || getIcon(target));
        });
      })["catch"](logError);
    } else {
      return Promise.resolve(setText(ord, ''));
    }
  };

  /**
   * Gets the root node ord if set. Returns null if no property / rootNodeOrd.
   *
   * @returns {baja.Ord | null}
   */
  OrdChooser.prototype.$getRootNodeOrd = function () {
    var propertyOrd = this.properties().getValue(ROOT_NODE_PROPERTY);
    if (propertyOrd) {
      return propertyOrd;
    } else {
      return null;
    }
  };

  //TODO: c.f. LinkEditor: can we root the picker at a different node other than 'local'?
  //e.g. "pick a node somewhere in the services container"
  /**
   * Resolve the root node to use to instantiate the component chooser tree.
   *
   * @private
   * @returns {Promise} promise to be resolved with the component
   * instance at which to root the tree
   */
  OrdChooser.prototype.$resolveTreeRootNode = function () {
    var offline = baja.isOffline();
    var that = this;
    var rootTreeOrd = this.$getRootNodeOrd();
    function makeNodeFromOrd(ord) {
      var makeTargetSpace;
      if (ord.toString().indexOf("file:") !== -1) {
        makeTargetSpace = stationTree.makeFileSpaceNode;
      } else if (ord.toString().indexOf("hierarchy:") !== -1) {
        makeTargetSpace = stationTree.makeHierarchySpaceNode;
      } else if (ord.toString().indexOf("history:") !== -1) {
        makeTargetSpace = stationTree.makeHistorySpaceNode;
      }
      if (makeTargetSpace) {
        return Promise.all([makeTargetSpace(), ordToPath(ord)]).then(function (_ref) {
          var _ref2 = _slicedToArray(_ref, 2),
            spaceRoot = _ref2[0],
            path = _ref2[1];
          return spaceRoot.getDescendent(path.slice(2, path.length));
        });
      } else {
        return makeOrd(ord).get().then(function (node) {
          return new BajaComplexTreeNode(node);
        });
      }
    }
    function makeStationNode() {
      return that.getOrdBase().then(resolveStation).then(function (station) {
        return new BajaComplexTreeNode(station);
      });
    }
    return Promise.all([(offline || !rootTreeOrd) && makeStationNode(), !offline && !rootTreeOrd && stationTree.makeFileSpaceNode(), !offline && !rootTreeOrd && stationTree.makeHierarchySpaceNode(), !offline && !rootTreeOrd && stationTree.makeHistorySpaceNode(), !offline && rootTreeOrd && makeNodeFromOrd(rootTreeOrd)]).then(function (nodes) {
      return stationTree.makeSessionNode(_.compact(nodes));
    });
  };

  /**
   * One-time instantiation of the nav tree to use as a node picker.
   * @private
   * @returns {Promise}
   */
  OrdChooser.prototype.$initializeNavTree = function () {
    var that = this;
    return that.$resolveTreeRootNode().then(function (rootNode) {
      return stationTree.createNavTree({
        value: rootNode,
        dom: that.$getNavTreeElement(),
        displayFilter: that.properties().getValue(DISPLAY_FILTER_PROPERTY)
      });
    });
  };

  /**
   * Ensure that the nav tree is loaded and ready to display. Will resolve
   * the tree root, instantiate and load the component chooser tree, and
   * pre-select a node in the tree to reflect the currently loaded ORD.
   *
   * @private
   * @param {baja.Ord} [ord]
   * @returns {Promise} promise to be resolved once the tree has loaded
   * and has a node pre-selected
   */
  OrdChooser.prototype.$loadOrdIntoNavTree = function (ord) {
    var that = this,
      invalidOrd,
      targetOrd = DEFAULT_ORD;
    if (isHierarchyOrd(ord) || isHistoryOrd(ord)) {
      targetOrd = ord;
    } else if (shouldResolve(ord)) {
      targetOrd = ord.resolve({
        base: baja.station
      }).then(getTargetOrd)["catch"](function () {
        invalidOrd = true;
        return DEFAULT_ORD;
      });
    } else {
      try {
        targetOrd = relativize(ord);
      } catch (ignore) {
        //invalid ORD, so use DEFAULT
      }
    }
    var rootOrd = that.$getRootNodeOrd();
    return Promise.all([Promise.resolve(targetOrd).then(ordToPath), that.$initializeNavTree(), rootOrd && ordToPath(rootOrd)]).then(function (_ref3) {
      var _ref4 = _slicedToArray(_ref3, 3),
        path = _ref4[0],
        tree = _ref4[1],
        rootOrdParts = _ref4[2];
      if (rootOrdParts) {
        var rootElementIndex = rootOrdParts.length - 1;
        path.splice(1, rootElementIndex - 1);
      }
      return tree.setSelectedPath(path, {
        silent: true
      })["catch"](function () {
        invalidOrd = true;
      });
    }).then(function () {
      return invalidOrd ? that.$setText(ord, '') : that.$commitSelectedNode();
    });
  };

  /**
   * Hides the nav tree (but does not unload it).
   *
   * @private
   */
  OrdChooser.prototype.$hideNavTree = function () {
    this.$getNavTreeElement().hide();
  };

  /**
   * Hides/shows the nav tree depending on whether it is currently shown.
   *
   * @private
   * @returns {Promise} promise to be resolved once the nav tree has
   * been toggled
   */
  OrdChooser.prototype.$toggleNavTree = function (ord) {
    var that = this;
    return that.$loadOrdIntoNavTree(ord).then(function () {
      that.$getNavTreeElement().toggle();
    });
  };

  /**
   * Read the ORD from the selected node in the tree and write it back to the
   * OrdChooser's text field.
   *
   * @private
   * @returns {Promise.<baja.Ord|null>} resolves the newly written ORD, or null
   * if no node was selected
   */
  OrdChooser.prototype.$commitSelectedNode = function () {
    var that = this,
      selectedNodes = that.$getNavTree().getSelectedNodes();
    if (selectedNodes.length) {
      var node = selectedNodes[0],
        ord = nodeToOrd(node, that.$getRootNodeOrd());
      return that.$updateDisplay(ord, node.getIcon()).then(_.constant(ord));
    }
    return Promise.resolve(null);
  };

  ////////////////////////////////////////////////////////////////
  // OrdChooser bajaux implementation
  ////////////////////////////////////////////////////////////////

  OrdChooser.prototype.requestFocus = function () {
    BaseEditor.prototype.requestFocus.apply(this, arguments);
    this.$selectAll();
  };

  /**
   * Builds up the ORD chooser DOM and arms event handlers for the change button
   * and nav tree events.
   *
   * @param {JQuery} dom
   */
  OrdChooser.prototype.doInitialize = function (dom) {
    var that = this;
    var selectFilterProperty = this.properties().getValue(SELECT_FILTER_PROPERTY);
    if (selectFilterProperty) {
      this.validators().add(selectFilterProperty);
    }
    dom.html(tplOrdChooser()).addClass('OrdChooser').toggleClass('hideOrdText', that.properties().getValue('showOrdText') === false);
    dom.on(NavTree.SELECTED_EVENT, '.tree', function () {
      that.$commitSelectedNode()["catch"](logError)["finally"](function () {
        that.setModified(true);
      });
    });
    dom.on(NavTree.ACTIVATED_EVENT, '.tree', function () {
      that.$commitSelectedNode().then(function (ord) {
        if (ord) {
          return that.validate().then(function () {
            dom.trigger(VALUE_READY_EVENT, [ord]);
            that.$hideNavTree();
          })["catch"](function (err) {
            that.setModified(true);
            throw err;
          });
        }
      })["catch"](logError);
    });
    dom.on('input', '.ord', function () {
      that.setModified(true);
      return false;
    });
    selectOnFocus(that, 'input', function () {
      return that.$selectAll();
    });
    return Promise.all([new IconEditor().initialize(that.$getIconElement()), fe.buildFor({
      dom: dom.children('.edit').children('.buttons'),
      value: that.getCommandGroup(),
      type: CommandButtonGroup
    })]);
  };

  /**
   * Interrogates the loaded ORD and updates display elements with the
   * relevant information.
   *
   * @param {baja.Ord|string} value
   * @returns {Promise} promise to be resolved when all display elements
   * are updated
   */
  OrdChooser.prototype.doLoad = function (value) {
    var that = this;
    if (that.getFormFactor() === Widget.formfactor.compact) {
      return that.$loadOrdIntoNavTree(makeOrd(value)).then(function () {
        that.$getNavTreeElement().show();
      });
    } else {
      return that.$updateDisplay(makeOrd(value));
    }
  };

  /**
   * Resolves the currently selected ORD.
   *
   * @returns {baja.Ord} the currently selected `baja.Ord`.
   */
  OrdChooser.prototype.doRead = function () {
    return makeOrd(this.$getOrdElement().val());
  };

  /**
   * Removes `OrdChooser` class and destroys child widgets.
   */
  OrdChooser.prototype.doDestroy = function () {
    this.jq().removeClass('OrdChooser hideOrdText');
    return Promise.all([this.$getIconEditor().destroy(), this.$getCommandButtonGroup().destroy()]);
  };

  /**
   * Enables/disables change button.
   *
   * @param readonly
   */
  OrdChooser.prototype.doReadonly = function (readonly) {
    return Promise.all([this.$getOrdElement().prop('readonly', readonly), this.$getChangeCommand().setEnabled(this.isEnabled() && !readonly), this.getChildWidgets().setAllReadonly(readonly)]);
  };

  /**
   * Enables/disables change button.
   *
   * @param enabled
   */
  OrdChooser.prototype.doEnabled = function (enabled) {
    return Promise.all([this.$getOrdElement().prop('disabled', !enabled), this.$getChangeCommand().setEnabled(!this.isReadonly() && enabled), this.getChildWidgets().setAllEnabled(enabled)]);
  };
  return OrdChooser;
});
