/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/SetDisplayNameCommand
 */
define(['baja!', 'Promise', 'underscore', 'lex!webEditors', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (baja, Promise, _, lexs, feDialogs, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0],
    DISPLAY_NAMES_FLAGS = baja.Flags.HIDDEN | baja.Flags.READONLY | baja.Flags.OPERATOR;

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Put all properties in the object in the NameMap.
   *
   * @inner
   * @param {Object} obj
   * @param {baja.NameMap} nameMap
   */
  function addAll(obj, nameMap) {
    _.each(nameMap.list(), function (key) {
      obj[key] = nameMap.get(key);
    });
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * A command for setting the display name on a slot on an editor's loaded
   * `Component` value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/SetDisplayNameCommand
   * @param {baja.Component} component the component on which to set slot
   * display names
   * @throws {Error} if no `Widget` provided
   */
  var SetDisplayNameCommand = function SetDisplayNameCommand(component) {
    var that = this;
    ComponentEditorCommand.call(that, {
      module: 'webEditors',
      lex: 'commands.setDisplayName'
    }, component);
  };
  SetDisplayNameCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  SetDisplayNameCommand.prototype.constructor = SetDisplayNameCommand;

  /**
   * Make sure we have admin write permissions.
   *
   * @param {baja.Component} comp
   * @returns {Boolean} true if I can set slot display names on this component
   */
  SetDisplayNameCommand.prototype.canPerformCommand = function (comp) {
    return comp.getPermissions().hasAdminWrite();
  };

  //TODO: support search/replace for multiple display names
  /**
   * Set the display names for the given slots.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots the slots for which to set display names
   * (note: only the first slot will actually get its display name set)
   * @returns {Promise} promise to be resolved after the user as entered
   * a new display name and it has been set on the component slot
   */
  SetDisplayNameCommand.prototype.performCommand = function (comp, slots) {
    var slot = slots[0];
    if (!slot) {
      return Promise.reject(new Error('slot to rename required'));
    }
    var oldDisplayName = comp.getDisplayName(slot);
    return feDialogs.showFor({
      title: webEditorsLex.get('commands.setDisplayName.description'),
      value: oldDisplayName,
      formFactor: 'mini'
    }).then(function (newDisplayName) {
      if (newDisplayName !== null && newDisplayName !== oldDisplayName) {
        return SetDisplayNameCommand.setDisplayName(comp, slot, newDisplayName);
      }
    });
  };

  /**
   * Performs the display name set by adding it to the component's
   * `displayNames` slot. If an empty string is given, the display name will
   * be removed.
   *
   * @param {baja.Component} comp
   * @param {String|baja.Slot} slot
   * @param {String} newDisplayName
   * @returns {Promise}
   */
  SetDisplayNameCommand.setDisplayName = function (comp, slot, newDisplayName) {
    var arg = {},
      displayNames = comp.get('displayNames'),
      method;
    if (displayNames) {
      method = 'set';
      addAll(arg, displayNames);
    } else {
      method = 'add';
    }
    if (newDisplayName) {
      arg[slot] = newDisplayName;
    } else {
      delete arg[slot];
    }
    return comp[method]({
      slot: 'displayNames',
      value: baja.NameMap.make(arg),
      flags: DISPLAY_NAMES_FLAGS //ignored in set()
    });
  };
  return SetDisplayNameCommand;
});
