/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/DeleteCommand
 */
define(['baja!', 'lex!webEditors', 'dialogs', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/commands/ComponentEditorCommand'], function (baja, lexs, dialogs, Promise, _, feDialogs, ComponentEditorCommand) {
  'use strict';

  var webEditorsLex = lexs[0];

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Remove all the given slots from the component, batching into one network
   * call.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot|String>} slots
   * @returns {Promise}
   */
  function removeAll(comp, slots) {
    var batch = new baja.comm.Batch(),
      removes = _.map(slots, function (slot) {
        return comp.remove({
          slot: slot,
          batch: batch
        });
      });
    batch.commit();
    return Promise.all(removes);
  }

  //TODO: optional confirmation dialog
  //TODO: found bug: add slot, rename, and delete all without refreshing multisheet. wrong slot goes away

  /**
   * A command for deleting slots from an editor's `Component` value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   * @alias module:nmodule/webEditors/rc/wb/commands/DeleteCommand
   * @param {baja.Component} component the component from which to delete slots
   * @throws {Error} if no `Widget` provided
   */
  var DeleteCommand = function DeleteCommand(component) {
    ComponentEditorCommand.call(this, {
      module: 'webEditors',
      lex: 'commands.delete'
    }, component);
  };
  DeleteCommand.prototype = Object.create(ComponentEditorCommand.prototype);
  DeleteCommand.prototype.constructor = DeleteCommand;

  /**
   * Make sure we have admin write permissions and we're not deleting a frozen
   * slot.
   *
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean} true if I can delete this slot
   */
  DeleteCommand.prototype.canPerformCommand = function (comp, slot) {
    return comp.getPermissions().hasAdminWrite() && !slot.isFrozen();
  };

  /**
   * Deletes the selected slots from the editor's loaded Component.
   *
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots the slots to delete
   * @returns {Promise}
   */
  DeleteCommand.prototype.performCommand = function (comp, slots) {
    return dialogs.showOkCancel({
      title: webEditorsLex.get('commands.delete.confirm.title'),
      content: webEditorsLex.getSafe('commands.delete.confirm.content')
    }).ok(function () {
      return removeAll(comp, slots)["catch"](feDialogs.error);
    }).promise();
  };
  return DeleteCommand;
});
