/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
 */
define(['baja!', 'bajaux/commands/Command', 'Promise', 'underscore'], function (baja, Command, Promise, _) {
  'use strict';

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////
  function isComponent(comp) {
    return baja.hasType(comp, 'baja:Component');
  }
  function finishValidate(cmd, result, updateEnabled) {
    if (updateEnabled) {
      cmd.setEnabled(result);
    }
    return result;
  }
  function getCompAndSlots(comp, slots) {
    if (!baja.hasType(comp, 'baja:Component')) {
      return;
    }
    var resultComp, resultSlots, parent, i;
    if (slots) {
      resultSlots = [];
      resultComp = comp;
      for (i = 0; i < slots.length; i++) {
        if (comp.has(slots[i])) {
          resultSlots.push(comp.getSlot(slots[i]));
        } else {
          return;
        }
      }
    } else {
      parent = comp.getParent();
      if (parent) {
        resultComp = parent;
        resultSlots = [comp.getPropertyInParent()];
      } else {
        resultComp = comp;
        resultSlots = [];
      }
    }
    if (resultComp && resultSlots) {
      return {
        comp: resultComp,
        slots: resultSlots
      };
    }
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * A command for working with an editor that can have a `Component` loaded.
   *
   * Why should you use this class? It is specifically implemented so that
   * it can be invoked on either a child slot of the loaded component, or
   * on the component itself. Imagine a Rename command - the version on the
   * MultiSheet would rename a child slot of the loaded component, while the
   * version on the NavTree would rename the selected component itself.
   *
   * It also include facilities for validating that it can only be invoked on
   * certain types of components or slots, updating its own enabled status
   * as appropriate.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand
   *
   * @param {Object} params Parameters object to pass to the `Command`
   * constructor. Note that if you instantiate a `ComponentEditorCommand` by
   * passing in a `func` parameter as you would an ordinary `Command`, it will
   * be *thrown away*. Implement `performCommand` instead.
   *
   * @param {baja.Component} component the component to invoke this command on.
   * This can be updated later with `setComponent`.
   *
   * @throws {Error} if no `Widget` provided
   */
  var ComponentEditorCommand = function ComponentEditorCommand(params, component) {
    var that = this;
    Command.call(that, _.extend({}, params, {
      enabled: false,
      /**
       * Invokes the command on the currently loaded component. Can be invoked
       * in one of two ways:
       *
       * - Pass a `slots` parameter to invoke the command on child slots of the
       *   loaded component. `performCommand` will receive the loaded component,
       *   and the `slots` parameter directly.
       * - Omit the `slots` parameter to invoke the command on the loaded
       *   component itself. `performCommand` will receive the parent component,
       *   and a `slots` array of length 1 with the loaded component's parent
       *   property.
       *
       * These two styles of invocation - "invoke on me" vs. "invoke on my kids"
       * - allow `performCommand` to be implemented once and work the same in
       * both situations.
       *
       * Do not attempt to implement `invoke` by passing a `func` parameter to
       * the constructor - override `performCommand` instead.
       *
       * @alias module:nmodule/webEditors/rc/wb/commands/ComponentEditorCommand#invoke
       * @param {Object} params
       * @param {Array.<String|baja.Slot>} [params.slots] specify which child
       * slots to invoke this command on
       * @returns {Promise} promise to be resolved after invocation, or
       * rejected if invocation fails, the component and slots could not be
       * determined from the loaded component and parameters, or if any
       * requested slots fail validation
       */
      func: function func(params) {
        params = params || {};
        var component = that.$component,
          obj = getCompAndSlots(component, params.slots);
        if (!obj) {
          return Promise.reject(new Error('could not determine component and slots'));
        }
        var comp = obj.comp,
          slots = obj.slots,
          badSlot = _.find(slots, function (slot) {
            return !that.canPerformCommand(comp, slot);
          });
        if (badSlot) {
          return Promise.reject(new Error('cannot perform command for slot ' + badSlot));
        }
        return that.performCommand(comp, slots, params);
      }
    }));
    that.setComponent(component);
  };
  ComponentEditorCommand.prototype = Object.create(Command.prototype);
  ComponentEditorCommand.prototype.constructor = ComponentEditorCommand;

  /**
   * Get the currently loaded component.
   *
   * @returns {baja.Component}
   */
  ComponentEditorCommand.prototype.getComponent = function () {
    return this.$component;
  };

  /**
   * Set the currently loaded component. If this is not an actual component,
   * the command will be immediately disabled.
   *
   * @param {baja.Component} comp
   */
  ComponentEditorCommand.prototype.setComponent = function (comp) {
    var that = this;
    if (!isComponent(comp)) {
      that.setEnabled(false);
      comp = null;
    }
    that.$component = comp;
  };

  /**
   * Check to see if this command can be successfully invoked with the currently
   * loaded component and requested slots. The slots are considered invalid
   * if they are empty, if any do not actually exist on the loaded component,
   * or if `canPerformCommand` fails for any slot.
   *
   * Optionally, enables/disables the command to match the validation result.
   *
   * @param {Array.<baja.Slot|String>} slots
   * @param {Boolean} updateEnabled if `true`, the command will be enabled
   * if validation succeeds or disabled if it fails.
   * @returns {Boolean} true if slots are valid
   */
  ComponentEditorCommand.prototype.validateSlots = function (slots, updateEnabled) {
    var that = this,
      comp = this.$component,
      slot,
      i;
    if (!comp || !slots.length) {
      return finishValidate(that, false, updateEnabled);
    }
    for (i = 0; i < slots.length; i++) {
      slot = comp.getSlot(slots[i]);
      if (!slot || !that.canPerformCommand(comp, slot)) {
        return finishValidate(that, false, updateEnabled);
      }
    }
    return finishValidate(that, true, updateEnabled);
  };

  /**
   * Override this to perform the actual work of the command.
   *
   * @abstract
   * @param {baja.Component} comp the component on which to invoke the command
   * @param {Array.<baja.Slot>} slots the slots on which to perform work
   * @param {Object} params params passed to the `invoke` call
   * @returns {Promise}
   */
  ComponentEditorCommand.prototype.performCommand = function (comp, slots, params) {};

  /**
   * Return true to indicate that this command can be successfully invoked
   * on the given slot. For example, do permissions checks on the component,
   * make sure the slot is not frozen, etc.
   *
   * @abstract
   * @param {baja.Component} comp
   * @param {baja.Slot} slot
   * @returns {Boolean}
   */
  ComponentEditorCommand.prototype.canPerformCommand = function (comp, slot) {
    return true;
  };
  return ComponentEditorCommand;
});
