/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/AddSlotEditor
 */
define(['baja!', 'lex!webEditors', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/wb/FlagConfig', 'nmodule/webEditors/rc/wb/PropertySheet'], function (baja, lexs, Promise, typeUtils, feDialogs, FlagConfig, PropertySheet) {
  'use strict';

  var webEditorsLex = lexs[0],
    getUniqueSlotName = typeUtils.getUniqueSlotName,
    importTypes = typeUtils.importTypes,
    toTypeSpec = typeUtils.toTypeSpec,
    lastUsedTypeSpec = 'baja:String';

  /**
   * Get the last type spec entered by the user.
   *
   * @inner
   * @returns {baja.Value} `baja:TypeSpec` instance
   */
  function getLastUsedTypeSpec() {
    return toTypeSpec(lastUsedTypeSpec);
  }

  /**
   * The "add slot" dialog just shows a property sheet for the name, typeSpec,
   * and flags fields. This just creates a dummy Component to load into the
   * property sheet.
   *
   * @inner
   * @param {baja.Component} targetComponent the component we're adding a slot
   * to
   * @param {String} [name] the default slot name. If none given, a default slot
   * name will be derived from the given type spec. If no type spec, the name
   * field will just be empty.
   * @param {String} [typeSpec] the type spec we want for the added slot. If
   * given, will be validated/imported via BajaScript.
   * @param {String} [display] the desired display type for the add slot
   * dialog
   * @returns {Promise} promise to be resolved with the dummy component
   * to show, or rejected if an invalid type spec given
   */
  function makeAddSlotComponent(targetComponent, name, typeSpec, display) {
    var simple = display === 'simple';
    return importTypes(typeSpec).then(function () {
      var comp = baja.$('baja:Component', {
        name: name || getUniqueSlotName(targetComponent, typeSpec),
        type: typeSpec ? toTypeSpec(typeSpec) : getLastUsedTypeSpec(),
        flags: 0,
        displayNames: baja.NameMap.make({
          name: webEditorsLex.get({
            key: 'SlotSheet.name',
            def: 'Name'
          }),
          type: webEditorsLex.get({
            key: 'SlotSheet.type',
            def: 'Type'
          }),
          flags: webEditorsLex.get({
            key: 'SlotSheet.flags',
            def: 'Flags'
          })
        })
      });
      if (simple) {
        comp.setFlags({
          slot: 'type',
          flags: baja.Flags.HIDDEN
        });
        comp.setFlags({
          slot: 'flags',
          flags: baja.Flags.HIDDEN
        });
      }
      return comp;
    });
  }

  //TODO: refactor, or add to build
  /**
   * Helper editor for adding a new slot.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/wb/commands/AddSlotEditor
   * @extends module:nmodule/webEditors/rc/wb/PropertySheet
   */
  var AddSlotEditor = function AddSlotEditor(params) {
    PropertySheet.apply(this, arguments);
  };
  AddSlotEditor.prototype = Object.create(PropertySheet.prototype);
  AddSlotEditor.prototype.constructor = AddSlotEditor;

  /**
   * Allow editing of `name`, `type`, and `flags` slots on a component.
   * @returns {Array}
   */
  AddSlotEditor.prototype.getSlotFilter = function () {
    return [{
      slot: 'name'
    }, {
      slot: 'type',
      properties: {
        showAbstract: false,
        showInterface: false,
        targetType: 'baja:Value'
      }
    }, {
      slot: 'flags',
      type: FlagConfig,
      properties: {
        dynamic: true
      }
    }];
  };

  /**
   * Reads the user-entered values and returns an object literal with
   * `slotName`, `value`, and `flags` properties.
   * @returns {Object}
   */
  AddSlotEditor.prototype.doRead = function () {
    return this.getChildEditors().readAll().then(function (values) {
      var name = values[0],
        type = values[1],
        flags = values[2];
      return {
        slotName: baja.SlotPath.escape(name),
        value: baja.$(type),
        flags: flags
      };
    });
  };

  /**
   * Make a throwaway component to show in an "Add Slot" dialog. Just has
   * simple name, type, and flags properties, with display names loaded from
   * the webEditors lexicon.
   *
   * @private
   * @param {Object} params
   * @param {String} [params.name] default slot name
   * @param {baja.Value} [params.value] value to add
   * @param {baja.Component} targetComponent the component to which we wish to
   * add a new slot (no changes will be made by this function)
   * @returns {Promise} promise to be resolved when the user has
   * entered values and clicked OK or Cancel
   */
  AddSlotEditor.doDialog = function (params, targetComponent) {
    params = params || {};
    var display = params.display,
      name = params.name,
      value = params.value,
      valueType = baja.hasType(value) && value.getType(),
      typeSpec = params.typeSpec || valueType && valueType.getTypeSpec();
    if (display === 'simple' && !typeSpec) {
      return Promise.reject(new Error('typeSpec param required if display is "simple"'));
    }
    return makeAddSlotComponent(targetComponent, name, typeSpec, display).then(function (comp) {
      //TODO: would be nice to auto-generate a slot name based on selected type
      return feDialogs.showFor({
        properties: {
          display: display,
          showHeader: false,
          showControls: false,
          showFooter: false
        },
        value: comp,
        title: webEditorsLex.get('commands.addSlot.displayName'),
        type: AddSlotEditor
      }).then(function (result) {
        if (result) {
          if (valueType) {
            result.value = value;
          }
          lastUsedTypeSpec = String(result.value.getType());
        }
        return result;
      });
    });
  };
  return AddSlotEditor;
});
