/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/AddSlotCommand
 */
define(['baja!', 'bajaux/commands/Command', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, Command, Promise, _, compUtils, typeUtils) {
  'use strict';

  var bulkAdd = compUtils.bulkAdd,
    promptForAddParameters = compUtils.promptForAddParameters,
    isComponent = typeUtils.isComponent;

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  function isFrozenSlot(comp, slotName) {
    var slot = comp.getSlot(slotName);
    return slot && slot.isFrozen();
  }
  function getDynamicSlotNames(comp) {
    return _.map(comp.getSlots().dynamic().toArray(), String);
  }

  /**
   * Reorder the dynamic slots of the component after adding a slot, to ensure
   * that the added slot is at the desired index in the component's slot list.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Object} params
   * @param {Array.<String>} params.slots the slots to move
   * @param {String} [params.before] move the new slots to be just before this
   * one
   * @param {String} [params.after] move the new slots to be just after this one
   * (only one of `before` or `after` will be used)
   * @returns {Promise} promise to be resolved after the reorder is finished
   */
  function doReorder(comp, params) {
    var slots = params.slots,
      before = params.before,
      after = params.after,
      slotName = String(before || after);
    if (slots && slots.length) {
      slots = _.map(slots, String);
      var allDynamicSlots = getDynamicSlotNames(comp),
        unmovedDynamicSlots = _.difference(allDynamicSlots, slots);
      if (isFrozenSlot(comp, slotName)) {
        return comp.reorder({
          dynamicProperties: slots.concat(unmovedDynamicSlots)
        });
      }
      var idx = _.indexOf(unmovedDynamicSlots, slotName);
      if (idx !== -1) {
        if (after) {
          idx++;
        }
        var slotsBefore = unmovedDynamicSlots.slice(0, idx),
          slotsAfter = unmovedDynamicSlots.slice(idx),
          toReorder = slotsBefore.concat(slots).concat(slotsAfter);
        return comp.reorder({
          dynamicProperties: toReorder
        });
      }
    } else {
      return Promise.resolve();
    }
  }
  function isAnnotation(value) {
    return baja.hasType(value, 'baja:WsAnnotation');
  }

  /**
   * Do the slot add.
   *
   * @inner
   * @param {baja.Component} comp
   * @param {Object} params
   * @param {String} params.slotName
   * @param {baja.Value} params.value
   * @param {Number} params.flags
   * @param {Object} [params.properties]
   * @returns {Promise}
   */
  function doAdd(comp, params) {
    var value = params.value,
      properties = params.properties;
    if (isComponent(value)) {
      value = value.newCopy(true);
      _.each(properties, function (valueToAdd, slotName) {
        if (slotName === 'wsAnnotation' && isAnnotation(valueToAdd)) {
          var existingAnno = value.get(slotName);
          if (isAnnotation(existingAnno)) {
            valueToAdd = valueToAdd.merge(existingAnno);
          }
        }
        value[value.has(slotName) ? 'set' : 'add']({
          slot: slotName,
          value: valueToAdd
        });
      });
    }
    return comp.add({
      slot: params.slotName,
      value: value,
      flags: params.flags
    });
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * A command for adding a new slot to an editor's `Component` value.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/commands/AddSlotCommand
   * @param {baja.Component} component the component to invoke this command on
   * @throws {Error} if no `Widget` provided
   */
  var AddSlotCommand = function AddSlotCommand(component) {
    var that = this;
    Command.call(that, {
      module: 'webEditors',
      lex: 'commands.addSlot',
      func: function func(params) {
        var comp = that.$component;
        if (!that.canPerformCommand(comp)) {
          throw new Error('cannot perform command');
        }
        return that.performCommand(comp, [], params || {});
      }
    });
    that.setComponent(component);
  };
  AddSlotCommand.prototype = Object.create(Command.prototype);
  AddSlotCommand.prototype.constructor = AddSlotCommand;
  AddSlotCommand.prototype.setComponent = function (component) {
    this.setEnabled(this.canPerformCommand(component));
    this.$component = component;
  };

  /**
   * Require ADMIN_WRITE permissions.
   *
   * @param {baja.Component} comp
   * @returns {Boolean} true if I have admin write permissions on this component
   */
  AddSlotCommand.prototype.canPerformCommand = function (comp) {
    return isComponent(comp) && comp.getPermissions().hasAdminWrite();
  };

  /**
   * Either adds a new slot to the loaded component directly, or shows an Add
   * Slot dialog, then adds a new slot to the commands's currently loaded
   * component using the data the user entered.
   *
   * @see module:nmodule/webEditors/rc/fe/baja/util/compUtils.promptForAddParameters
   * @param {baja.Component} comp
   * @param {Array.<baja.Slot>} slots
   * @param {Object} params
   * @param {String} [params.slotName] pass this and `value` to add a new slot
   * without prompting the user
   * @param {baja.Value} [params.value] pass this and `slotName` to add a new
   * slot without prompting the user
   * @param {String} [params.before] pass this to specify that the
   * added slot should go immediately before this one
   * @param {String} [params.after] pass this to specify that the
   * added slot should go immediately after this one
   * @param {Array.<baja.Value>} [params.bulkValues] pass an array of values
   * to add an arbitrary number of values; user will be prompted for a slot
   * name only if exactly one value is being added
   * @param {Array.<String>} [params.names] if using `bulkValues`, optionally
   * use this to pre-specify slot names to use
   * @param {Object} [params.properties] if adding a Component, specify a map
   * of baja Values to be set or added as Properties on that Component
   * @returns {Promise}
   * @see module:nmodule/webEditors/rc/fe/baja/util/compUtils#bulkAdd
   *
   * @example
   *   <caption>Add a number of new values with specified slot names.</caption>
   *   addSlotCommand.invoke({
   *     bulkValues: [
   *       baja.$('control:NumericPoint'),
   *       baja.$('control:NumericPoint'),
   *       baja.$('control:NumericPoint')
   *     ],
   *     names: [ 'point1', 'point2', 'point3' ]
   *   });
   *
   * @example
   *   <caption>Add a number of new values; slot names will be auto-generated
   *   from the values' type specs.</caption>
   *   addSlotCommand.invoke({
   *     bulkValues: [
   *       baja.$('control:NumericPoint'),
   *       baja.$('control:StringPoint')
   *     ]
   *   });
   *
   * @example
   *   <caption>Add a value with a specified slot name (user will not be
   *   prompted)</caption>
   *   addSlotCommand.invoke({
   *     slotName: 'myNewPoint',
   *     value: baja.$('control:NumericPoint'),
   *     properties: { wsAnnotation: baja.$('baja:WsAnnotation', '1,2,3,4') }
   *   });
   *
   * @example
   *   <caption>Add a value, specifying which slot the new slot should go
   *   after.</caption>
   *   addSlotCommand.invoke({
   *     slotName: 'larrySlot',
   *     value: 'Larry Fine',
   *     after: 'moeSlot'
   *   });
   *
   * @example
   *   <caption>Add a value, prompting the user for a slot name.</caption>
   *   addSlotCommand.invoke({
   *     value: baja.$('control:NumericPoint')
   *   });
   */
  AddSlotCommand.prototype.performCommand = function (comp, slots, params) {
    var toAddTo = slots[0] ? comp.get(slots[0]) : comp,
      bulkValues = params.bulkValues;
    if (params.slotName && params.value) {
      return doAdd(toAddTo, params);
    }
    if (bulkValues) {
      return bulkAdd(toAddTo, bulkValues, {
        names: params.names
      }).then(function (addedSlots) {
        return doReorder(toAddTo, {
          slots: addedSlots,
          before: params.before,
          after: params.after
        });
      });
    }
    return promptForAddParameters(toAddTo, params).then(function (result) {
      if (!result) {
        return;
      }
      _.extend(result, _.pick(params, 'properties'));
      return doAdd(toAddTo, result).then(function () {
        return doReorder(toAddTo, {
          slots: [result.slotName],
          before: params.before,
          after: params.after
        });
      });
    });
  };
  return AddSlotCommand;
});
