/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/wb/commands/ActionsCommand
 */
define(['baja!', 'lex!webEditors', 'bajaux/commands/Command', 'nmodule/webEditors/rc/fe/feDialogs', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/wb/ActionFirer'], function (baja, lexs, Command, feDialogs, BaseEditor, ActionFirer) {
  'use strict';

  var webEditorsLex = lexs[0];

  /**
   * Slot does not have HIDDEN slot flag.
   * @inner
   * @param {baja.Slot} slot
   * @returns {boolean}
   */
  function notHidden(slot) {
    return !(slot.getFlags() & baja.Flags.HIDDEN);
  }

  /**
   * Return true if the value is a mounted components with at least one
   * Action slot.
   *
   * @inner
   * @param {baja.Component} comp
   * @returns {boolean}
   */
  function hasMountedActions(comp) {
    return baja.hasType(comp, 'baja:Component') && comp.isMounted() && comp.getSlots().actions().filter(notHidden).toArray().length > 0;
  }

  /**
   * Editor has a value loaded that is capable of firing actions (either a
   * mounted Component with non-hidden Actions, or an Action itself).
   *
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/baja/BaseEditor} ed
   * @returns {boolean}
   */
  function canFireActions(ed) {
    var value = ed.value();
    if (hasMountedActions(value)) {
      return true;
    } else if (ed.isComplexSlotEditor() && value instanceof baja.Action && notHidden(value) && hasMountedActions(ed.getComplex())) {
      return true;
    }
    return false;
  }
  function isBaseEditor(ed) {
    return ed instanceof BaseEditor;
  }

  /**
   * A command for firing Actions on an editor's `Component` value.
   *
   * Will enable itself if the editor has a `Component` loaded with at least
   * one Action slot to fire, or if the editor is a `ComplexSlotEditor` on
   * a mounted Component with an Action slot loaded directly.
   *
   * @class
   * @extends module:bajaux/commands/Command
   * @alias module:nmodule/webEditors/rc/wb/commands/ActionsCommand
   * @param {module:nmodule/webEditors/rc/fe/baja/BaseEditor} ed the
   * `BaseEditor` to bind this `Command` to
   * @throws {Error} if no `BaseEditor` provided
   */
  var ActionsCommand = function ActionsCommand(ed) {
    if (!isBaseEditor(ed)) {
      throw new Error('BaseEditor required');
    }
    var that = this;
    Command.call(that, {
      module: 'webEditors',
      lex: 'commands.actions',
      enabled: canFireActions(ed),
      /**
       * Shows an `ActionFirer` dialog, if appropriate.
       *
       * @alias module:nmodule/webEditors/rc/wb/commands/ActionsCommand#invoke
       * @see module:nmodule/webEditors/rc/wb/ActionFirer
       */
      func: function func() {
        if (!canFireActions(ed)) {
          return;
        }
        var value = ed.value();
        if (value instanceof baja.Action) {
          return feDialogs.action({
            component: ed.getComplex(),
            slot: value
          });
        }
        return ActionFirer.showActionDialog({
          component: value,
          title: webEditorsLex.get('commands.actions.description')
        });
      }
    });

    /**
     * Allows this Command to update its own enabled status based on the value
     * currently loaded in the watched `BaseEditor`. If actions can be fired
     * on the editor's loaded value, this command will be enabled.
     *
     * @private
     * @memberOf module:nmodule/webEditors/rc/wb/commands/ActionsCommand
     */
    function $updateEnabled() {
      that.setEnabled(canFireActions(ed));
    }
    ed.on('loaded', $updateEnabled);
    that.$updateEnabled = $updateEnabled;
  };
  ActionsCommand.prototype = Object.create(Command.prototype);
  ActionsCommand.prototype.constructor = ActionsCommand;
  return ActionsCommand;
});
