/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*jshint forin: false */
/*global localStorage, sessionStorage */

define([], function () {
  'use strict';

  var _local, _session, _localStorage;
  try {
    _localStorage = localStorage;
  } catch (ignore) {}

  /**
   * @inner
   * @param {Object} storage
   * @returns {boolean}
   */
  function storageWorks(storage) {
    var x = '__webEditors';
    try {
      storage.setItem(x, x);
      storage.removeItem(x);
      return true;
    } catch (e) {
      return false;
    }
  }

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/util/storageUtil
   */
  var exports = {};

  /**
   * Return true if a valid `localStorage` exists.
   *
   * @returns {boolean}
   */
  exports.hasLocalStorage = function () {
    return storageWorks(_localStorage);
  };

  /**
   * Return the local storage object to use. If a valid `localStorage` object
   * cannot be found, one will be shimmed in (note that it will be cleared
   * when the page is closed or refreshed).
   *
   * @returns {Object}
   */
  exports.getLocalStorage = function () {
    if (!_local) {
      _local = exports.hasLocalStorage() ? _localStorage : exports.$polyfill();
    }
    return _local;
  };

  /**
   * Return true if a valid `sessionStorage` exists.
   *
   * @returns {boolean}
   */
  exports.hasSessionStorage = function () {
    return storageWorks(sessionStorage);
  };

  /**
   * Return the session storage object to use. If a valid `sessionStorage`
   * object cannot be found, one will be shimmed in (note that it will be
   * cleared when the page is closed or refreshed).
   *
   * @returns {Object}
   */
  exports.getSessionStorage = function () {
    if (!_session) {
      _session = exports.hasSessionStorage() ? sessionStorage : exports.$polyfill();
    }
    return _session;
  };

  //TODO: this should really be some OSS shim but probably too late to get approved at this point
  //http://dev.w3.org/html5/webstorage/#storage-0
  /**
   * Create a new Storage polyfill object to use when a valid
   * `localStorage`/`sessionStorage` object cannot be found.
   *
   * @private
   * @returns {Object}
   */
  exports.$polyfill = function () {
    var obj = {};
    Object.defineProperty(obj, 'length', {
      enumerable: false,
      get: function get() {
        return Object.keys(this).length;
      }
    });
    Object.defineProperty(obj, 'getItem', {
      enumerable: false,
      value: function value(key) {
        if (arguments.length === 0) {
          throw new Error('1 argument required');
        }
        if (this.hasOwnProperty(String(key))) {
          return this[key];
        }
        return null;
      }
    });
    Object.defineProperty(obj, 'setItem', {
      enumerable: false,
      value: function value(key, _value) {
        if (arguments.length < 2) {
          throw new Error('2 arguments required');
        }
        this[key] = String(_value);
      }
    });
    Object.defineProperty(obj, 'removeItem', {
      enumerable: false,
      value: function value(key) {
        if (arguments.length === 0) {
          throw new Error('1 argument required');
        }
        delete this[key];
      }
    });
    Object.defineProperty(obj, 'clear', {
      enumerable: false,
      value: function value() {
        var that = this;
        Object.keys(that).forEach(function (prop) {
          delete that[prop];
        });
      }
    });
    Object.defineProperty(obj, 'key', {
      enumerable: false,
      value: function value(idx) {
        if (arguments.length === 0) {
          throw new Error('1 argument required');
        }
        return Object.keys(this)[idx] || null;
      }
    });
    return obj;
  };
  return exports;
});
