/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Shivani Jakhmola
 */

define([], function () {
  'use strict';

  /**
   * @API Status: **Private**
   *
   * @since Niagara 4.8
   * @exports nmodule/webEditors/rc/util/byteUtils
   */
  var byteUtils = {};

  /**
   * Function is used to append zeros to the string if it does not contain two values.
   *
   * @param {String} stringFromId - string value
   * @param {number} displayLength - the number of values required in the string ,
   * as per the radix value
   * @param {String} stringBytes - string containing byte values
   * @returns {String} string with zeros
   */
  function getStringOfZeros(stringFromId, displayLength) {
    var paddedValue = stringFromId;
    for (var n = stringFromId.length; n < displayLength; ++n) {
      paddedValue = "0".concat(paddedValue);
    }
    return paddedValue;
  }

  /**
   * Function is used to add zeros at the end of the byte array if it does not contain
   * the required number of elements
   *
   * @param {Array.<number>} idByteArray - byte array containing domain id
   * @param {number} len- length of the domain id allowed
   */
  function addTrailingZeros(idByteArray, len) {
    for (var i = idByteArray.length; i < len; i++) {
      idByteArray[i] = 0;
    }
    return idByteArray;
  }

  /**
   * Function is used to convert byte array to string.
   *
   * @example
   * //creates a string converting all the values to hex equivalent.
   * byteArrayToString([0x32,0x10,0xa,0x0,0x1,0x4],6)
   * //creates a string "20.0a.0a.00.01.7f"
   *
   * @example
   * //creates a string converting all the values to hex equivalent when length
   * of id provided is lesser than the expected len
   * byteArrayToString([ 100, 20 ],3) //creates a string "64.14"
   *
   *
   * @param {Array.<number>} byteArray - byte array containing the id .It holds hex values
   * @param {string} delimiter - delimiter character like [, . ; :]
   * @param {number} len - acceptable length of the domain id like 0,1,3,6 etc
   * @param radix
   * @returns {string} byte array string including the delimiter
   */
  byteUtils.byteArrayToString = function (byteArray, delimiter, len, radix) {
    var stringBytes = ""; //holds the byte array as string with delimiter
    var displayLength = 0x00ff.toString(radix.valueOf()).length; // gets a value 2
    //limiting the domain id length to the acceptable value of len
    if (len > byteArray.length) {
      len = byteArray.length;
    }

    //return an empty string if len is given as 0
    if (len === 0) {
      return "";
    }
    for (var i = 0; i < len; i++) {
      // Add delimiter before all bytes except first.
      if (i > 0) {
        stringBytes = stringBytes + delimiter;
      }
      // Convert byte to string with hex conversion
      var stringFromId = (byteArray[i] & 0x00ff).toString(radix.valueOf());
      var paddedValue = getStringOfZeros(stringFromId, displayLength);
      stringBytes = stringBytes.concat(paddedValue);
    }
    return stringBytes.toString();
  };

  /**
   * Function is used to convert string to byte array.
   * String can be delimited with space : , . ; / \
   * @param {string} string - string containing the values to be made into a
   * byte array .It holds hex values.
   * @param {number} len - acceptable length of the domain id like 0,1,3,6 etc
   * @param {string} [params.delimiter] - delimiter used to separate the string
   * @param {number} [params.radix] - radix used for number conversion.
   * @returns {Array.<number>|null} byte array
   */
  byteUtils.stringToByteArray = function (string, len) {
    var _ref = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {},
      _ref$delimiter = _ref.delimiter,
      delimiter = _ref$delimiter === void 0 ? /[:,. ;/\\]/ : _ref$delimiter,
      _ref$radix = _ref.radix,
      radix = _ref$radix === void 0 ? 16 : _ref$radix;
    if (len === 0) {
      return [];
    }

    /* checking if id contains odd number of elements and converting
     it to even number of elements by prepending 0 */

    if ((string.length & 0x01) > 0) {
      string = "0" + string;
    }
    var idSplit = string.split(delimiter);
    var tokenCount = idSplit.length;
    // Handle special case of hex bytes with no delimiter
    if (tokenCount === 1 && radix === 16) {
      var byteArray = string.match(/..?/g).map(function (s) {
        return parseInt(s, radix);
      }); //for id ="25665" it returns [ 2, 86, 101 ]
      return addTrailingZeros(byteArray, len);
    } else if (len < 0) {
      len = tokenCount;
    }

    /* assigns idByteArray with values from idSplit after converting to
    hex values and discarding the NaN values which might get generated
    when string contains extra delimiter "34::45:::" */

    var idByteArray = idSplit.slice(0, len).map(function (s) {
      return parseInt(s, radix);
    }).filter(function (f) {
      return !isNaN(f);
    });

    // add zeros in the end
    addTrailingZeros(idByteArray, len);
    return idByteArray;
  };
  return byteUtils;
});
