function _typeof(o) { "@babel/helpers - typeof"; return _typeof = "function" == typeof Symbol && "symbol" == typeof Symbol.iterator ? function (o) { return typeof o; } : function (o) { return o && "function" == typeof Symbol && o.constructor === Symbol && o !== Symbol.prototype ? "symbol" : typeof o; }, _typeof(o); }
/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/*eslint-env browser */ /*jshint browser: true */

define(['log!nmodule.webEditors.rc.util.SyncedSessionStorage', 'Promise', 'underscore', 'nmodule/webEditors/rc/util/storageUtil', 'nmodule/js/rc/asyncUtils/asyncUtils', 'nmodule/js/rc/polyfills/BroadcastChannel'], function (log, Promise, _, storageUtil, asyncUtils, BroadcastChannel) {
  'use strict';

  var getLocalStorage = storageUtil.getLocalStorage,
    getSessionStorage = storageUtil.getSessionStorage,
    waitInterval = asyncUtils.waitInterval,
    logWarning = log.warning.bind(log),
    STORE_KEY = 'SyncedSessionStorage';

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/util/SyncedSessionStorage
   */
  var exports = {};
  var channel;
  var SyncedSessionStorage = function SyncedSessionStorage() {};

  //how long to wait for other tabs to report their data. a background tab in
  //chrome can take up to a second, so wait this long.
  //TODO: we may need to revisit this again in M57. https://groups.google.com/a/chromium.org/forum/#!msg/blink-dev/XRqy8mIOWps/LvBUbWbxAQAJ
  exports.INIT_DELAY = 1000;

  /**
   * @param {String} key
   * @returns {String}
   */
  SyncedSessionStorage.prototype.getItem = function (key) {
    return getValue(key);
  };

  /**
   * Store a key/value pair in session storage and sync it with
   *  other tabs.
   *
   * @param {string} key
   * @param {string} value
   */
  SyncedSessionStorage.prototype.setItem = function (key, value) {
    // Make sure key is a string
    if (typeof key !== 'string') {
      throw new Error('SyncedSessionStorage.setItem key must be string');
    }
    // Make sure value is a string
    if (typeof value !== 'string') {
      throw new Error('SyncedSessionStorage.setItem value must be string');
    }
    setValue(key, value, +new Date());
    return value;
  };
  SyncedSessionStorage.prototype.clear = function () {
    clear();
  };

  //TODO: key, removeItem, clear

  function getStorage() {
    return isWb() ? getLocalStorage() : getSessionStorage();
  }
  function parseStore(data) {
    try {
      var store = JSON.parse(data);
      if (store && _typeof(store.data) === 'object' && store.user === getUser()) {
        return store;
      }
    } catch (ignore) {}
  }
  function retrieveStore() {
    var storage = getStorage(),
      data = storage.getItem(STORE_KEY),
      store = parseStore(data);
    if (store && isStillValid(store)) {
      return store;
    }
    store = defaultStore();
    storage.setItem(STORE_KEY, JSON.stringify(store));
    return store;
  }
  function getValue(key) {
    var data = retrieveStore().data;
    return key in data ? data[key] : null;
  }
  function setValue(key, value, timestamp) {
    var store = retrieveStore();
    store.data[key] = value;
    if (timestamp) {
      store.timestamp = timestamp;
    }
    store.user = getUser();
    getStorage().setItem(STORE_KEY, JSON.stringify(store));
  }
  function createInstance() {
    var instance = new SyncedSessionStorage(),
      setItem = instance.setItem,
      clear = instance.clear;
    if (isWb()) {
      if (retrieveStore().user !== getUser()) {
        getStorage().setItem(STORE_KEY, JSON.stringify(defaultStore()));
      }
      return Promise.resolve(instance);
    }

    //TODO: i haven't a clue how to go about testing the cross-tab initialization. iframes?
    var initResponses = [];
    if (channel) {
      channel.close();
    }
    channel = new BroadcastChannel('SyncedSessionStorage');
    channel.onmessage = function (event) {
      var data = event.data;
      if (data === 'requestAll') {
        channel.postMessage({
          isInitResponse: true,
          store: retrieveStore()
        });
      } else if (data === 'clear') {
        clear();
      } else if (data && data.isInitResponse) {
        initResponses.push(data.store);
      } else {
        return setValue(data.key, data.value);
      }
    };
    instance.setItem = function (key, value) {
      setItem.apply(this, arguments);
      channel.postMessage({
        key: key,
        value: value
      });
    };
    instance.clear = function () {
      clear.apply(this, arguments);
      channel.postMessage('clear');
    };

    //to initialize, we're going to request data from all other tabs that
    //have SyncedSessionStorage running, and use whatever is most recent.

    //request all data via channel - this will populate the response array
    channel.postMessage('requestAll');
    return waitInterval(exports.INIT_DELAY).then(function () {
      try {
        var user = getUser();
        //get the most recent data that matches our current user...
        var latest = _.reduce(initResponses, function (memo, store) {
          return store.user === user && store.timestamp > memo.timestamp ? store : memo;
        }, retrieveStore());
        if (latest.user !== user) {
          latest = defaultStore();
        }
        //and store it.
        getStorage().setItem(STORE_KEY, JSON.stringify(latest));
        return instance;
      } catch (e) {
        logWarning('could not initialize synced session storage - defaulting to native');
        logWarning(e);
        return getStorage();
      }
    });
  }
  function isStillValid(store) {
    return !(isWb() && isPriorToWorkbenchBoot(store.timestamp));
  }
  function isPriorToWorkbenchBoot(timestamp) {
    return timestamp < getNreBootTime();
  }
  function getEnv() {
    return window.niagara && window.niagara.env || {};
  }
  function getUser() {
    return getEnv().user;
  }
  function getNreBootTime() {
    return getEnv().nreBootTime;
  }
  function isWb() {
    return getEnv().type === 'wb';
  }
  function clear() {
    getStorage().setItem(STORE_KEY, JSON.stringify(defaultStore()));
  }
  function defaultStore() {
    return {
      data: {},
      timestamp: -1,
      user: getUser()
    };
  }

  /**
   * Gets an object matching the `SessionStorage` API that will sync across
   * browser tabs.
   *
   * @function
   * @returns {Promise}
   */
  exports.getInstance = _.once(createInstance);

  /**
   * @private
   * @function
   */
  exports.$createInstance = createInstance;
  return exports;
});
