/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*eslint-env browser */ /*jshint browser: true */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/util/ListSelection
 */
define(['jquery', 'underscore', 'nmodule/js/rc/tinyevents/tinyevents'], function ($, _, tinyevents) {
  'use strict';

  /**
   * Manage user selection of different elements in an array, such as table
   * rows, icons, files in an explorer-like interface, etc. Handles toggling
   * and selecting swaths of elements (ctrl-click and shift-click,
   * respectively).
   *
   * @class
   * @mixes tinyevents
   *
   * @param {Boolean} params.supportSwipeSelect true if this ListSelection should select
   *  when touch is swiped; if false then swiping can be used to scroll.
   *
   * @param {Number} params.longPressDelay milliseconds to delay before activating long-press,
   *  which will select an item (where press toggles an item).
   *
   * @param {Function} params.longPressHandler
   *
   * @constructor
   * @alias module:nmodule/webEditors/rc/util/ListSelection
   */
  var ListSelection = function ListSelection(params) {
    var that = this;
    that.$bits = []; //TODO: maybe more performant way of doing this?
    that.$anchor = 0;
    this.$supportSwipeSelect = params && params.supportSwipeSelect;
    this.$longPressDelay = params && params.longPressDelay || 500;
    this.$longPressHandler = params && params.longPressHandler;
    tinyevents(that);

    /**
     * A default jQuery click handler - will get the index of the element on
     * which it was triggered, check for ctrl/shift keys, and select elements as
     * appropriate.  Use this handler if the list contains only one DOM element per item.
     *
     // * If the list entries each contain more than one element, see
     * `ListSelection#installTouchHandler`.
     *
     * @param {JQuery.Event} e
     * @example
     *   var selection = new ListSelection();
     *   dom.on('click', '.row', selection.defaultHandler);
     *   //later
     *   dom.off('click', selection.defaultHandler);
     */
    that.defaultHandler = function (e) {
      var idx = $(this).index();
      if (e.which !== 3 || !that.isSelected(idx)) {
        that[e.shiftKey ? 'swath' : e.ctrlKey ? 'toggle' : 'select'](idx);
      }
    };

    /**
     * A default jQuery click / touch handler - works the same as the default handler
     * but also supports touchstart / touchend / touchmove events.
     *
     * When an item is touched, it is toggled, unless long-press is used, in which case
     * the item is selected.
     *
     * Any time the item is touched and the touch element (stylus or finger) is moved,
     * long-press is disabled and any item that is touched is selected, and previously
     * selected items are not unselected.
     *
     * @param {JQuery.Event} e
     * @example
     *   var selection = new ListSelection();
     *   dom.on('click touchstart touchend touchmove', '.row', selection.defaultTouchHandler);
     *   //later
     *   dom.off('click touchstart touchend touchmove', selection.defaultTouchHandler);
     */
    that.defaultTouchHandler = function (e) {
      var idx = $(this).index();
      switch (e.type) {
        case 'touchstart':
          return that.$touchStartHandler(idx, e);
        case 'touchend':
          e.preventDefault();
          return that.$touchEndHandler(idx, e);
        case 'touchmove':
          return that.$touchMoveHandler(idx, e);
        case 'click':
          e.preventDefault();
          if (e.which !== 3 || !that.isSelected(idx)) {
            return that[e.shiftKey ? 'swath' : e.ctrlKey ? 'toggle' : 'select'](idx);
          }
      }
    };
    this.setExclusiveFilter();
  };

  /**
   * Install correct touch handlers for the specified dom, container selector and child selector;
   * Publishes changes events to the changeCallback after events have settled.
   *
   * This is a convenience function for handling a DOM where the rows of the list contain more than
   * one element, such as in a multi-column table.
   *
   * @param {JQuery} dom - Primary DOM (usually this.jq())
   * @param {String} containerSelector - Container selector, such as '.ux-table'
   * @param {String} childSelector - Child selector, such as '.ux-table-row'.  The elements
   *    specified by this selector must be direct children of containerSelector element.
   * @param {Function} [changeCallback] - Called after changes to the model have occurred
   */
  ListSelection.prototype.installTouchHandler = function (dom, containerSelector, childSelector, changeCallback) {
    var that = this;
    if (that.$installed) {
      throw new Error('ListSelection#installTouchHandler can only be called once for any given ListSelection.');
    }
    that.$installed = true;
    function getEffectiveTarget(e) {
      var target;
      // touchmove does not always have the target being touched, which is what is required
      if (e.type === 'touchmove') {
        var x, y;
        if (e.touches && e.touches.length > 0) {
          x = e.touches[0].clientX;
          y = e.touches[0].clientY;
        } else {
          // Try to stay as backwards compatible as possible
          x = e.clientX;
          y = e.clientY;
        }
        target = $(document.elementFromPoint(x, y));
        if (!target) {
          target = $(e.target);
        }
      } else {
        target = $(e.target);
      }
      if (target) {
        target = target.closest(childSelector).get(0);
      }
      // Ensure the target was found, and is a child of this editor
      if (target && dom.find(containerSelector).get(0) === $(target).closest(containerSelector).get(0)) {
        return target;
      }
    }
    function handleTouches(e) {
      var target = getEffectiveTarget(e);
      if (target) {
        that.defaultTouchHandler.apply(target, arguments);
      }
    }
    dom.on('click touchstart touchend touchmove', childSelector, handleTouches);
    // Defer the changeCallback call until all of the events have been handled, to
    // ensure ListSelection overrides any default browser behavior (such as automatically
    // checking input checkboxes).
    that.on('changed', function () {
      window.setTimeout(function () {
        if (changeCallback) {
          changeCallback();
        }
      }, 0);
    });
  };

  /**
   *
   * @param {function} filter function that takes an index.  This function should return
   *  true if the element at the index is an exclusive element (i.e. when it is in a
   *  selected state, no other no other elements be in a selected state).
   *
   *  When an exclusive element is selected, all other elements are deselected.
   *  When a non-exclusive element is selected, all exclusive elements are deselected.
   */
  ListSelection.prototype.setExclusiveFilter = function (filter) {
    this.$exclusiveFilterFunction = filter || function () {
      return false;
    };
  };
  ListSelection.prototype.$setTimeout = function (f) {
    var that = this;
    that.$timeout = setTimeout(function () {
      that.$isTimerRunning = false;
      return f();
    }, that.$longPressDelay);
  };
  ListSelection.prototype.$clearTimeout = function () {
    this.$isTimerRunning = false;
    clearTimeout(this.$timeout);
  };
  ListSelection.prototype.$startTimer = function (fn) {
    this.$isTimerRunning = true;
    this.$mouseDownTime = new Date().getTime();
    return this.$setTimeout(fn);
  };
  ListSelection.prototype.$isTimedOut = function () {
    var pressTime = new Date().getTime() - this.$mouseDownTime;
    return pressTime > this.$longPressDelay;
  };
  ListSelection.prototype.$touchStartHandler = function (idx, e) {
    var that = this;
    that.$previousIdx = idx;
    that.$startTimer(function () {
      if (that.$longPressHandler) {
        that.$longPressHandler(e, idx);
      } else {
        // Long press selects the item and deselects the rest
        that.select(idx);
      }
    });
  };
  ListSelection.prototype.$touchEndHandler = function (idx, e) {
    if (this.$isTimerRunning) {
      this.$clearTimeout();
      this.toggle(idx);
    }
    this.$previousIdx = -1;
  };
  ListSelection.prototype.$touchMoveHandler = function (idx, e) {
    var changed;
    // Once the finger is moved, don't consider longpress
    // until finger is lifted and pressed again.
    this.$clearTimeout();
    if (this.$supportSwipeSelect) {
      e.preventDefault();
      if (idx !== this.$previousIdx && this.$previousIdx !== -1) {
        if (!this.$exclusiveFilterFunction(this.$previousIdx) && !this.isSelected(this.$previousIdx)) {
          this.$resetExclusives();
          this.put(this.$previousIdx, true);
          changed = true;
        }
      }
      if (!this.$exclusiveFilterFunction(idx) && !this.isSelected(idx)) {
        this.$resetExclusives();
        this.put(idx, true);
        changed = true;
      }
      this.$previousIdx = idx;
      if (changed) {
        this.emit('changed');
      }
    }
  };

  /**
   * Return true if the given index is selected.
   *
   * @param {Number} idx
   * @returns {Boolean}
   */
  ListSelection.prototype.isSelected = function (idx) {
    return !!this.$bits[idx];
  };

  /**
   * Sets the selection to only this row / rows. Call this when user clicks
   * with no modifiers held down.
   *
   * @param {Number} idx the index to select
   * @param {Number} [end] pass a second number to select a contiguous row
   * (end-exclusive)
   */
  ListSelection.prototype.select = function (idx, end) {
    var oldBits = this.$bits,
      bits = this.$bits = [],
      changed,
      i;
    if (idx >= 0) {
      end = end || idx + 1;
      for (i = 0; i < idx; i++) {
        if (oldBits[i]) {
          changed = true;
        }
      }
      for (i = idx; i < end; i++) {
        if (!oldBits[i]) {
          changed = true;
        }
        bits[i] = true;
      }
      for (i = end; i < oldBits.length; i++) {
        if (oldBits[i]) {
          changed = true;
        }
      }
    } else {
      for (i = 0; i < oldBits.length; i++) {
        if (oldBits[i]) {
          changed = true;
        }
      }
    }
    this.$anchor = idx;
    if (changed) {
      this.emit('changed');
    }
  };

  /**
   * Silently set the selected status of the specified index/indices. Will not
   * trigger a `changed` event.
   *
   * @param {number|Array.<number>} idx
   * @param {boolean} selected
   */
  ListSelection.prototype.put = function (idx, selected) {
    if (typeof idx === 'number') {
      idx = [idx];
    }
    selected = !!selected;
    for (var i = 0; i < idx.length; i++) {
      this.$bits[idx[i]] = selected;
    }
  };

  /**
   * Toggles the selected status of this row. Call this when user clicks with
   * Ctrl key held down.
   * @param {Number} idx the index to toggle
   */
  ListSelection.prototype.toggle = function (idx) {
    // If this index is exclusive, use select rather than toggle.
    if (this.$exclusiveFilterFunction(idx)) {
      return this.select(idx);
    }
    var bits = this.$bits;
    bits[idx] = !bits[idx];
    this.$anchor = idx;
    this.$resetExclusives();
    this.emit('changed');
  };
  ListSelection.prototype.$resetExclusives = function () {
    var that = this;
    that.$bits.forEach(function (value, index) {
      if (value && that.$exclusiveFilterFunction(index)) {
        that.$bits[index] = false;
      }
    });
  };

  /**
   * Selects a contiguous group of elements. Call this when user clicks with
   * Shift key held down. The last element that was previously selected or
   * toggled will be the start of the group and the given index will be the end.
   * Previously selected elements are not unselected.
   * @param {Number} idx
   */
  ListSelection.prototype.swath = function (idx) {
    // If idx item is exclusive, select it
    if (this.$exclusiveFilterFunction(idx)) {
      return this.select(idx);
    }
    var anchor = this.$anchor,
      bits = this.$bits,
      start = Math.min(anchor, idx),
      end = Math.max(anchor, idx),
      changed;
    for (var i = start; i <= end; i++) {
      if (this.$exclusiveFilterFunction(i)) {
        // If the anchor is exclusive and selected, deselect it
        if (i === anchor) {
          if (bits[i]) {
            changed = true;
            bits[i] = false;
          }
        }
        // Skip the other exclusive items.
        continue;
      }
      if (!bits[i]) {
        changed = true;
      }
      bits[i] = true;
    }
    if (changed) {
      this.emit('changed');
    }
  };

  /**
   * Insert a block of new rows at a specific index.
   *
   * @param {number} index first index to insert
   * @param {number} length the number of rows to insert
   * @param {boolean} [selected=false] whether the new rows should be selected
   */
  ListSelection.prototype.insert = function (index, length, selected) {
    var bits = this.$bits;
    if (bits.length < index) {
      bits[index] = false;
    }
    Array.prototype.splice.apply(bits, [index, 0].concat(_.map(_.range(0, length), _.constant(!!selected))));
    if (length) {
      this.emit('changed');
    }
  };

  /**
   * Remove rows from this selection. Either a contiguous range can be removed,
   * or an array of specific indices. Rows that come after the removed rows will
   * be shifted down appropriately.
   *
   * @param {number|Array.<number>} start index of first row to remove, or the
   * specified array of indices to be removed
   * @param {number} [end] remove rows up to but not including this index
   */
  ListSelection.prototype.remove = function (start, end) {
    var indices = _.isArray(start) ? start.slice() : _.range(start, end),
      bits = this.$bits,
      j = 0;
    if (!indices.length) {
      return;
    }

    // Remove indices that are out of range (otherwise indices.length is wrong).
    indices = _.filter(indices, function (index) {
      return index < bits.length;
    });
    indices.sort(byIndex);
    this.$bits = _.filter(bits, function (bit, i) {
      if (indices[j] === i) {
        j++;
        return false;
      } //remove it
      return true;
    }).slice(0, bits.length - indices.length);
    this.emit('changed');
  };

  /**
   * Clear the whole selection. Call this if the contents of the backing array
   * are wiped/reordered/etc.
   */
  ListSelection.prototype.clear = function () {
    this.$bits = [];
    this.emit('changed');
  };

  /**
   * Return a subset of the elements in the given array that correspond to
   * indexes that are currently in the selection.
   *
   * @param {Array} array
   * @returns {Array} array of selected elements
   */
  ListSelection.prototype.getSelectedElements = function (array) {
    var bits = this.$bits;
    return _.filter(array, function (elem, i) {
      return bits[i];
    });
  };

  /**
   * Return true if no elements are currently selected.
   *
   * @returns {Boolean}
   */
  ListSelection.prototype.isEmpty = function () {
    var bits = this.$bits;
    for (var i = 0, len = bits.length; i < len; i++) {
      if (bits[i]) {
        return false;
      }
    }
    return true;
  };
  function byIndex(a, b) {
    return a - b;
  }
  return ListSelection;
});
