/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'underscore', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/compUtils'], function (baja, _, Promise, compUtils) {
  'use strict';

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   *
   * RPC calls for working with server-side user data. All data will be
   * automatically keyed to the current user.
   *
   * @exports nmodule/webEditors/rc/servlets/userData
   */
  var exports = {};

  /**
   * Retrieve existing data for this key from the station.
   *
   * @param {string} key
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise.<string>}
   */
  exports.get = function (key, params) {
    return isInvalidKey(key) || doRpc('get', [key], params);
  };

  /**
   * Store a value on the station for this key.
   *
   * @param {string} key
   * @param {string} data
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise}
   */
  exports.put = function (key, data, params) {
    return isInvalidKey(key) || isInvalidData(data) || doRpc('put', [key, data], params);
  };

  /**
   * Delete the value on the station for this key.
   *
   * @param {string} key
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch]
   * @returns {Promise}
   */
  exports.del = function (key, params) {
    return isInvalidKey(key) || doRpc('delete', [key], params);
  };

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  function doRpc(method, args, params) {
    return baja.rpc({
      ord: 'service:web:UserDataConfig',
      method: method,
      args: args
    })["catch"](function (ignore) {
      return null;
    });
  }
  function isInvalidKey(key) {
    if (!key || typeof key !== 'string') {
      return Promise.reject(new Error('key required'));
    }
  }
  function isInvalidData(data) {
    if (typeof data !== 'string') {
      return Promise.reject(new Error('data required'));
    }
  }

  /**
   * Encodes all child values of an object to BSON objects.
   * @param {Object.<String, baja.Value>} valueMap
   * @return {Object.<String, object>}
   */
  exports.toPersistenceObject = function (valueMap) {
    if (!valueMap) {
      return {};
    } else {
      return _.mapObject(valueMap, function (value) {
        if (baja.hasType(value)) {
          return baja.bson.encodeValue(value);
        } else {
          return exports.toPersistenceObject(value);
        }
      });
    }
  };

  /**
   * Takes a component and a PersistenceObject and applies all pairs in the value map to
   * a new copy of the component. Only components with a predefined structure are supported; adding
   * Complex slots to a component without matching structure will not work.
   * @param {baja.Component} config
   * @param {object.<string, *>} [userPreferences]
   * @returns {Promise.<baja.Component>}
   */
  exports.fromPersistenceObject = function (config, userPreferences) {
    if (!userPreferences) {
      userPreferences = {};
    }
    var allSlotPromises = [];
    var newComponent = compUtils.newCopyComplete(config);
    _.each(userPreferences, function (value, slot) {
      var slobject = {
        slot: slot,
        value: value
      };
      if (config.get(slot) instanceof baja.Complex) {
        allSlotPromises.push(exports.fromPersistenceObject(newComponent.get(slot), value).then(function (comp) {
          newComponent.set({
            slot: slot,
            value: comp
          });
        }));
      } else if (config.has(slot)) {
        allSlotPromises.push(newComponent.set(slobject));
      } else {
        allSlotPromises.push(newComponent.add(slobject));
      }
    });
    return Promise.all(allSlotPromises).then(function () {
      return newComponent;
    });
  };
  return exports;
});
