/**
 * @copyright 2018 Tridium, Inc. All Rights Reserved.
 * @author Tony Richards
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/servlets/QueryAdapter
 */
define(['baja!', 'Promise', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/util/chunkUtil'], function (baja, Promise, tinyevents, chunkUtil) {
  'use strict';

  /**
   * QueryAdapter is an adapter for QueryServlet via chunkUtil.
   *
   * This adapter acts as a translation layer for the query, the query results,
   * and the model.
   *
   * @class
   * @mixes tinyevents
   * @alias module:nmodule/webEditors/rc/servlets/QueryAdapter
   */
  var QueryAdapter = function QueryAdapter() {
    tinyevents(this);
  };

  /**
   * Convert the query into a string, adding whatever additional arguments
   * are necessary to complete the query, and escaping it so that it can
   * be used as part of a URI.
   *
   * @param {Object} query
   * @param {baja.Ord} query.ord
   * @returns {string}
   */
  QueryAdapter.prototype.getFullOrdString = function (query) {
    return query.ord.toString();
  };

  /**
   * Reset the model.
   *
   * Override this with the implementation to reset your model.
   *
   * @returns {Promise}
   *
   * @abstract
   */
  QueryAdapter.prototype.resetModel = function () {
    throw new Error("resetModel() not implemented");
  };

  /**
   * Execute the query.
   *
   * @param {Object} query
   * @param {baja.Ord} query.ord
   *
   * @returns {Promise} resolves when the full query has been executed and
   *  the results have been returned.  Call #getRecords to get the results.
   */
  QueryAdapter.prototype.doQuery = function (query) {
    var that = this;
    return that.resetModel().then(function () {
      var dataUri = that.$getDataUri(query);
      var params = {};
      that.$undecodedRecords = [];
      params.checkAbort = function () {
        return that.$isStopped;
      };
      params.progress = function (chunks) {
        that.$undecodedRecords = that.$undecodedRecords.concat(chunks.map(JSON.parse));
        return that.$triggerProgressEvent();
      };
      return chunkUtil.ajax(dataUri, params).then(function () {
        if (that.$undecodedRecords.length) {
          // The first record can be a JsonObject with just a numeric lastPage entry so we know the lastPage that contains any data.
          var firstObject = that.$undecodedRecords[0];
          if (firstObject && typeof firstObject.lastPage === "number") {
            return that.$triggerChangePageEvent(firstObject.lastPage);
          }
          // First record
          return that.$decodeRecord(firstObject).then(function (firstRecord) {
            return that.$addColumnsFromRecord(firstRecord).then(function () {
              return that.addRecords([firstRecord]);
            });
          }).then(function () {
            // Subsequent records
            return Promise.all(that.$undecodedRecords.slice(1).map(function (record) {
              return that.$decodeRecord(record);
            }));
          }).then(function (records) {
            return that.addRecords(records);
          });
        } else {
          return that.$addColumnsFromRecord();
        }
      });
    });
  };

  /**
   * Called by the QueryServlet API to add records to the model.
   *
   * Override this with your implementation to add records to the model.
   *
   * @param {Array<Object>} records
   * @param {Number} [index] index to insert the rows; will append to the
   * end if omitted
   *
   * @returns {Promise} resolves after the record has been added to the model.
   *
   * @abstract
   */
  QueryAdapter.prototype.addRecords = function (records, index) {
    throw new Error("addRecords() not implemented");
  };

  /**
   * Get the records after doQuery has been called.
   *
   * Override this with your implementation of getting records from your model.
   *
   * @returns {Array}
   *
   * @abstract
   */
  QueryAdapter.prototype.getRecords = function () {
    throw new Error("getRecords() not implemented");
  };

  /**
   * Obtain the default record used for generating the columns. The first record is provided as an argument,
   * but you may want to override this method to return the default record type if you want to provide
   * a default set of columns when there is no first record available.
   *
   * @param {baja.Complex} [record] If there is no first column, then this will be undefined.
   * @returns {baja.Complex|Promise.<baja.Complex>|undefined}
   *
   */

  QueryAdapter.prototype.getDefaultRecord = function (record) {
    return record;
  };

  /**
   * Called by the QueryServlet API to add one or more columns to the model.
   *
   * Override this with your implementation to add a column to your model.
   *
   * @param {Array<baja.Property>} columns array of objects that represents the
   * meta data for the columns to add to the model.
   * @param {baja.Complex} parent `baja.Complex`.
   *
   * @returns {Promise} resolves after the column has been added to the model.
   *
   * @abstract
   */
  QueryAdapter.prototype.addColumns = function (columns, parent) {
    throw new Error("addColumns() not implemented");
  };

  /**
   * Get the columns
   *
   * Override this with your implementation of getting the columns of your model.
   *
   * @returns {Object} each property in this object is the name of a column, and
   * the value of the property is the meta data for that column (displayName, type)
   *
   * @abstract
   */
  QueryAdapter.prototype.getColumns = function () {
    throw new Error("getColumns() not implemented");
  };

  /**
   * Get the URI for the servlet, including the escaped ord.
   *
   * @param {Object} query
   * @param {baja.Ord} query.ord
   * @returns {string}
   *
   * @private
   */
  QueryAdapter.prototype.$getDataUri = function (query) {
    var fullOrd = this.getFullOrdString(query);
    return "/wsbox/query/data/" + fullOrd;
  };

  /**
   * Trigger a progress event.
   *
   * @private
   */
  QueryAdapter.prototype.$triggerProgressEvent = function () {
    return this.emit("progress");
  };

  /**
   * Trigger a change page event to change the current page.
   *
   * @param {Number} newPage
   * @returns {Array.<*>} the results of all the handler calls, or an empty
   * array if there are no handlers defined.
   * @private
   */
  QueryAdapter.prototype.$triggerChangePageEvent = function (newPage) {
    return this.emit(QueryAdapter.CHANGE_PAGE_EVENT, newPage);
  };

  /**
   * Decode a record response
   *
   * @param {Object} record as a BSON encoded object
   * @returns {Promise} resolves after the record has been decoded
   *
   * @private
   */
  QueryAdapter.prototype.$decodeRecord = function (record) {
    return baja.bson.decodeAsync(record);
  };

  /**
   * Determine the columns to add to the model based on the first row of the data
   * within a `baja.Complex` record.
   *
   * @param {baja.Complex} [record] If there is no first record, then this will be undefined.
   * @returns {Promise} resolves when the columns have been added to the model
   *
   * @private
   */
  QueryAdapter.prototype.$addColumnsFromRecord = function (record) {
    var that = this;
    return Promise.resolve(that.getDefaultRecord(record)).then(function (rec) {
      var columns = rec ? rec.getSlots().properties().toArray() : [];
      return that.addColumns(columns, rec);
    });
  };

  /** Triggered when a different page is requested. */
  QueryAdapter.CHANGE_PAGE_EVENT = 'changePage';
  return QueryAdapter;
});
