/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'baja!baja:UnitConversion', 'Promise', 'underscore'], function (baja, types, Promise, _) {
  'use strict';

  var UC = baja.$('baja:UnitConversion');

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  function getConversionOrdinal(unitConversion) {
    if (baja.hasType(unitConversion, 'baja:Enum')) {
      return unitConversion.getOrdinal();
    }
    if (baja.hasType(unitConversion, 'baja:Number')) {
      return unitConversion.valueOf();
    }
    if (typeof unitConversion === 'string') {
      if (!UC.getRange().isTag(unitConversion)) {
        throw new Error('invalid BUnitConversion tag ' + unitConversion);
      }
      return UC.get(unitConversion).getOrdinal();
    }
    return null;
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   *
   * Utilities for working with numbers.
   *
   * @exports nmodule/webEditors/rc/fe/baja/util/numberUtils
   */
  var exports = {};

  /**
   * Convert the number to a display string. Units will not be included in the
   * resultant string, but the number itself will still be converted if the
   * `units` and `unitConversion` params are present.
   *
   * @returns {Promise.<String>}
   */
  exports.toDisplay = function (num, params) {
    params = params || {};
    return exports.convertUnitTo(num, params.units, params.unitConversion).then(function (num) {
      return num.toString(_.omit(_.extend({
        precision: 2
      }, params), 'units'));
    });
  };

  /**
   * Convert the number to a display string, accounting for units and unit
   * conversion.
   *
   * @param {Number} num
   * @param {Object} params all params supported by `toDisplay()`
   * @param {baja.Unit} [params.units]
   * @param {baja.FrozenEnum|String|Number} [params.unitConversion]
   * @returns {Promise.<string>}
   */
  exports.toDisplayWithUnits = function (num, params) {
    //TODO: default precision to 2 in bajaScript Double impl. c.f. BDouble#toString.
    return Promise.resolve(num.toString(_.extend({
      precision: 2
    }, params)));
  };

  /**
   * Convert the given number in the specified unit, applying the given unit
   * conversion.
   *
   * @param {Number} num
   * @param {baja.Unit} unit the units the given number is considered to be in
   * @param {baja.FrozenEnum|String|Number} unitConversion the desired unit
   * system (English or metric)
   *
   * @example
   * <caption>I know I have 32 degrees Fahrenheit (English), but the user wants
   * to see metric units. What number should I show them?</caption>
   *
   * exports.convertUnitTo(32, fahrenheit, 'metric')
   *   .then(function (celsius) {
   *     expect(celsius).toBeCloseTo(0); //remember JS rounding inaccuracy
   *   });
   */
  exports.convertUnitTo = function (num, unit, unitConversion) {
    return exports.getDisplayUnits(unit, unitConversion).then(function (displayUnits) {
      return displayUnits ? num.make(unit.convertTo(displayUnits, num.valueOf())) : num;
    });
  };

  /**
   * Convert the given number in the specified unit, removing the given unit
   * conversion.
   *
   * @param {Number} num
   * @param {baja.Unit} unit the units we want to calculate for the given
   * number
   * @param {baja.FrozenEnum|String|Number} unitConversion the unit system that
   * the given number is in
   *
   * @example
   * <caption>The user has the UI configured to show Celsius (metric), and has
   * typed 0. But I know the underlying point is configured for Fahrenheit. What
   * "real" number should I write to the point?</caption>
   *
   * exports.convertUnitFrom(0, fahrenheit, 'metric')
   *   .then(function (fahrenheit) {
   *     expect(fahrenheit).toBeCloseTo(32); //remember JS rounding inaccuracy
   *   });
   */
  exports.convertUnitFrom = function (num, unit, unitConversion) {
    return exports.getDisplayUnits(unit, unitConversion).then(function (displayUnits) {
      return displayUnits ? num.make(displayUnits.convertTo(unit, num.valueOf())) : num;
    });
  };

  /**
   * Get the display unit specified by the given unit/conversion combination.
   *
   * @param {baja.Unit} [unit]
   * @param {baja.FrozenEnum|String|Number} [unitConversion]
   * @returns {Promise} promise to be resolved with the desired display unit,
   * or `null` if no unit given
   */
  exports.getDisplayUnits = function (unit, unitConversion) {
    if (!unit || unit.equals(baja.Unit.NULL)) {
      return Promise.resolve(null);
    }
    return baja.UnitDatabase.get().then(function (db) {
      unit = db.getUnit(unit.getUnitName() || unit) || unit;
      if (!unitConversion) {
        return unit;
      }
      var ordinal = getConversionOrdinal(unitConversion),
        tag = baja.$('baja:UnitConversion', ordinal).getTag();
      return db.convertUnit(tag, unit);
    });
  };
  return exports;
});
