/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

define(['baja!', 'baja!baja:Month',
//required for toDateTimeString
'bajaux/Properties', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/numberUtils'], function (baja, types, Properties, Promise, _, numberUtils) {
  'use strict';

  var toDisplayWithUnits = numberUtils.toDisplayWithUnits;

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /** @param {String} err */
  function reject(err) {
    return Promise.reject(new Error(err));
  }

  ////////////////////////////////////////////////////////////////
  // Exports
  ////////////////////////////////////////////////////////////////

  /**
   * API Status: **Private**
   * @exports nmodule/webEditors/rc/fe/baja/util/facetsUtils
   */
  var exports = {};

  /**
   * Resolve a string formatted representation of the given simple, using the
   * facets to format it in different ways.
   *
   * The following Simple types are supported:
   *
   * - `baja:Boolean`
   * - `baja:Double`
   * - `baja:DynamicEnum`
   * - `baja:Float`
   * - `baja:String`
   *
   * And the following facets are supported:
   *
   * - `falseText`
   * - `precision`
   * - `range`
   * - `trueText`
   * - `units`
   *
   * @param {baja.Simple} simple
   * @param {baja.Facets|Object} facets
   * @returns {Promise} promise to be resolved with a display string
   */
  exports.formatSimple = function (simple, facets) {
    if (baja.hasType(facets, 'baja:Facets')) {
      facets = facets.toObject();
    }
    facets = facets || {};
    var type = simple.getType();
    switch (type.toString()) {
      case 'baja:Double':
      case 'baja:Float':
        return toDisplayWithUnits(simple, facets);
      case 'baja:AbsTime':
      case 'baja:Boolean':
      case 'baja:Date':
      case 'baja:DynamicEnum':
      case 'baja:String':
      case 'baja:Time':
        return Promise.resolve(simple.toString(facets));
      default:
        return reject('unsupported Type ' + type);
    }
  };

  //TODO: covered by fe/BaseEditor tests, add self contained test
  /**
   * `baja.Facets.make` throws up if null/undefined keys are given. This will
   * filter out any values that are not valid to add to a Facets instance.
   *
   * @param {Object|baja.Facets} obj key-value mapping to make into a Facets.
   * If already a `baja.Facets` will be returned directly.
   * @returns {baja.Facets}
   */
  exports.toFacets = function (obj) {
    if (obj instanceof baja.Facets) {
      return obj;
    }
    var result = {},
      key,
      value;
    for (key in obj) {
      if (obj.hasOwnProperty(key)) {
        value = obj[key];
        if (baja.hasType(value)) {
          result[key] = value;
        }
      }
    }

    //TODO: should baja.Facets handle null/undefined?
    return baja.Facets.make(result);
  };

  /**
   * Return a new Facets instance, constructed from the input Facets, with
   * a single value added, changed, or removed. Since Facets are BSimples
   * and therefore immutable, we have this method to set one key/value pair
   * and return a new instance.
   *
   * @param {baja.Facets} facets the Facets instance on which to change a
   * key/value mapping
   * @param {String} key the key to change
   * @param {baja.Value|null|undefined} value the value to set on the new
   * instance. If null/undefined, the key will be removed completely.
   * @returns {baja.Facets} new Facets instance with the one key/value pair
   * changed
   */
  exports.setFacet = function (facets, key, value) {
    if (!facets) {
      throw new Error('facets required');
    }
    if (!key) {
      throw new Error('key required');
    }
    var obj = {},
      keys = facets.getKeys();
    if (value === null || value === undefined) {
      //remove facet
      _.each(keys, function (existingKey) {
        if (existingKey !== key) {
          obj[existingKey] = facets.get(existingKey);
        }
      });
    } else {
      _.each(keys, function (existingKey) {
        obj[existingKey] = facets.get(existingKey);
      });
      obj[key] = value;
    }
    return exports.toFacets(obj);
  };

  /**
   * Apply Baja Facets, as hidden bajaux Properties, to a Widget.
   *
   * @param {baja.Facets|object} facets
   * @param {module:bajaux/Widget} widget
   */
  exports.applyFacets = function (facets, widget) {
    var props = widget.properties();
    facets = exports.toFacets(facets || baja.Facets.DEFAULT);
    _.each(facets.getKeys(), function (key) {
      var value = facets.get(key);
      key = baja.SlotPath.unescape(key);
      if (props.get(key)) {
        props.setValue(key, value);
      } else {
        props.add({
          hidden: true,
          name: key,
          "transient": true,
          typeSpec: value.getType().getTypeSpec(),
          value: value
        });
      }
    });
  };

  /**
   * Convert a `baja.Facets` instance to a `bajaux/Properties` instance. All
   * values from the `Facets` will become hidden, transient `Properties`.
   *
   * @param {baja.Facets} facets
   * @returns {module:bajaux/Properties}
   */
  exports.toProperties = function (facets) {
    facets = exports.toFacets(facets);
    return new Properties(_.map(facets.getKeys(), function (key) {
      return {
        name: key,
        value: facets.get(key),
        hidden: true,
        "transient": true
      };
    }));
  };
  return exports;
});
