/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/util/IntegralHelper
 */
define(['baja!', 'Promise', 'bajaScript/baja/obj/numberUtil', 'nmodule/webEditors/rc/fe/baja/util/numberUtils'], function (baja, Promise, bajaNumberUtil, numberUtils) {
  'use strict';

  var toDisplay = numberUtils.toDisplay,
    getNumberCharacters = bajaNumberUtil.getNumberCharacters;

  /**
   * A helper class for working with integral values.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/baja/util/IntegralHelper
   * @param {Function} ctor constructor for the type of number you wish to
   * edit. Defaults to `baja.Long`.
   * @param {Object} [config]
   * @param {Number} [config.radix] radix to use when converting an integer
   * to string
   * @param {Boolean} [config.forceSign] always include the sign, even with
   * positive values
   * @param {Boolean} [config.showSeparators] include separators, as in "1,000"
   * @param {Number} [config.zeroPad] pad the number with zeroes to ensure at
   * least this many digits to the left of the decimal sign
   */
  var IntegralHelper = function IntegralHelper(ctor, config) {
    var instance = ctor && ctor.DEFAULT,
      type = typeof instance !== 'undefined' && instance.getType();
    if (!type || !type.is('baja:Integer')) {
      instance = baja.Long.DEFAULT;
    }
    this.$instance = instance;
    this.$config = config || {};
  };

  /**
   * Coerce a number to the expected type.
   * @param {baja.Integer|baja.Long} num
   * @returns {baja.Integer|baja.Long}
   */
  IntegralHelper.prototype.toType = function (num) {
    return this.$instance.make(+num);
  };

  //TODO: this resolves '2asdfasdf' to '2' due to parseInt.
  /**
   * Parses an integer string, respecting special encodings like `max`
   * and `radix` facet.
   *
   * @param {String} str the string to parse
   * @returns {baja.Integer|baja.Long} the parsed number
   */
  IntegralHelper.prototype.parse = function (str) {
    var that = this,
      instance = that.$instance,
      radix = str.toLowerCase().indexOf('0x') === 0 ? 16 : that.$getRadix() || 10,
      chars = getNumberCharacters(this.$config.languageTag || baja.getLanguage()),
      groups = new RegExp('\\' + chars.grouping, 'g'),
      num = parseInt((str || '0').replace(groups, ''), radix);
    if (isNaN(num)) {
      if (str === 'max' || str === 'min') {
        return instance.decodeFromString(str);
      } else {
        throw new Error('invalid numeric value ' + str);
      }
    }
    return this.toType(num);
  };

  /**
   * Formats a number into its string representation, respecting special
   * encodings like `max` and `radix` facet.
   *
   * @param {baja.Integer|baja.Long} number the number to format
   * @returns {Promise.<String>} the formatted string
   */
  IntegralHelper.prototype.format = function (number) {
    var config = this.$config,
      str = number.encodeToString();
    if (str === 'max' || str === 'min') {
      return Promise.resolve(str);
    }
    return toDisplay(number, {
      forceSign: config.forceSign,
      languageTag: config.languageTag,
      precision: 0,
      showSeparators: config.showSeparators,
      zeroPad: config.zeroPad,
      radix: this.$getRadix()
    });
  };
  IntegralHelper.prototype.$getRadix = function () {
    var radix = this.$config.radix;
    return radix < 2 || radix > 36 ? undefined : radix;
  };
  return IntegralHelper;
});
