/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/util/FloatingPointHelper
 */
define(['baja!', 'underscore', 'bajaScript/baja/obj/numberUtil', 'nmodule/webEditors/rc/fe/baja/util/numberUtils'], function (baja, _, bajaNumberUtil, numberUtils) {
  'use strict';

  var toDisplay = numberUtils.toDisplay,
    getNumberCharacters = bajaNumberUtil.getNumberCharacters;

  /**
   * A helper class for working with floating point values.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/baja/util/FloatingPointHelper
   * @param {Function} ctor constructor for the type of number you wish to
   * edit. Defaults to `baja.Double`.
   * @param {Object} [config]
   * @param {Number} [config.precision] how many decimal points to show
   * @param {Boolean} [config.forceSign] always include the sign, even with
   * positive values
   * @param {Boolean} [config.showSeparators] include separators, as in "1,000"
   * @param {Number} [config.zeroPad] pad the number with zeroes to ensure at
   * least this many digits to the left of the decimal sign
   */
  var FloatingPointHelper = function FloatingPointHelper(ctor, config) {
    var instance = ctor && ctor.DEFAULT,
      type = typeof instance !== 'undefined' && instance.getType();
    if (!type || !type.is('baja:Float')) {
      instance = baja.Double.DEFAULT;
    }
    this.$instance = instance;
    this.$config = config || {};
  };

  /**
   * Coerce a number to the expected type.
   * @param {Number|baja.Float} num
   * @returns {Number|baja.Float}
   */
  FloatingPointHelper.prototype.toType = function (num) {
    return this.$instance.make(+num);
  };

  /**
   * Parses a floating point string, respecting special encodings like `+inf`
   * and `precision` facet.
   *
   * @param {String} str the string to parse
   * @returns {baja.Double|baja.Float} the parsed number
   */
  FloatingPointHelper.prototype.parse = function (str) {
    var instance = this.$instance;
    var _getNumberCharacters = getNumberCharacters(this.$config.languageTag || baja.getLanguage()),
      grouping = _getNumberCharacters.grouping,
      decimal = _getNumberCharacters.decimal;
    var decimals = new RegExp('\\' + decimal, 'g');

    // treat all whitespace equally - i don't think a french keyboard has an nnbsp key
    var groups = grouping.match(/\s/) ? '\\s' : '\\' + grouping;

    // also accept grouping as opposite of decimal
    groups += '|' + (decimal === '.' ? ',' : '\\.');
    groups = new RegExp(groups, 'g');
    var num = +str.replace(groups, '').replace(decimals, '.');
    if (isNaN(num)) {
      if (str !== '-inf' && str !== '+inf' && str !== 'nan') {
        throw new Error('invalid numeric value ' + str);
      } else {
        return instance.decodeFromString(str);
      }
    }
    return this.toType(num);
  };

  /**
   * Formats a number into its string representation, respecting special
   * encodings like `+inf` and `precision`, `showSeparators`, and `forceSign`
   * facets.
   *
   * @param {baja.Double|baja.Float|Number} number the number to format
   * @returns {Promise.<String>} the formatted string
   */
  FloatingPointHelper.prototype.format = function (number) {
    return toDisplay(number, _.omit(this.$config, 'radix'));
  };
  return FloatingPointHelper;
});
