/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/util/DepthSubscriber
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.baja.util.DepthSubscriber', 'Promise', 'underscore'], function (baja, log, Promise, _) {
  'use strict';

  var logError = log.severe.bind(log);

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Convert the value to an array if it isn't one already.
   * @inner
   * @param {Array|*} arr
   * @returns {Array} the given array, or a single-element array containing the
   * input value
   */
  function toArray(arr) {
    return Array.isArray(arr) ? arr : [arr];
  }

  /**
   * Load all child components of the given array of components.
   * @inner
   * @param {Array.<baja.Component>} comps
   * @returns {Promise} promise to be resolved with a flattened array of
   * all child components of the given components
   */
  function getAllKids(comps) {
    return Promise.all(_.invoke(comps, 'loadSlots')).then(function () {
      return _.flatten(_.map(comps, function (comp) {
        return comp.getSlots().is('baja:Component').toValueArray();
      }));
    });
  }

  /**
   * Subscribe the entire component tree to the given depth.
   *
   * @inner
   * @param {Array.<baja.Component>} comps components to subscribe
   * @param {baja.Subscriber} sub the subscriber to use
   * @param {Number} depth the depth to subscribe - 0 subscribes nothing, 1
   * subscribes only the given components, 2 subscribes components and their
   * kids, etc.
   * @returns {Promise} promise to be resolved when the whole component
   * tree is subscribed
   */
  function subscribeAll(comps, sub, depth) {
    if (comps.length === 0 || depth === 0) {
      return Promise.resolve();
    }
    return Promise.resolve(sub.subscribe(comps)).then(function () {
      return getAllKids(comps);
    }).then(function (allKids) {
      return subscribeAll(allKids, sub, depth - 1);
    });
  }

  /**
   * Unsubscribe the entire component tree to the given depth.
   *
   * @inner
   * @param {Array.<baja.Component>} comps components to unsubscribe
   * @param {baja.Subscriber} sub the subscriber to use
   * @param {Number} depth the depth to unsubscribe - 0 unsubscribes nothing, 1
   * unsubscribes only the given components, 2 unsubscribes components and their
   * kids, etc.
   * @returns {Promise} promise to be resolved when the whole component
   * tree is unsubscribed
   */
  function unsubscribeAll(comps, sub, depth) {
    if (comps.length === 0 || depth === 0) {
      return Promise.resolve();
    }
    return getAllKids(comps).then(function (allKids) {
      return unsubscribeAll(allKids, sub, depth - 1);
    }).then(function () {
      return sub.unsubscribe(comps);
    });
  }

  ////////////////////////////////////////////////////////////////
  // DepthSubscriber
  ////////////////////////////////////////////////////////////////

  //TODO: add filtering capability, c.f. hack-around in BacnetDeviceUxManager
  /**
   * A wrapper for a `baja.Subscriber` that will subscribe/unsubscribe an entire
   * component tree.
   *
   * @param {Number} depth the depth to which to subscribe the component tree
   * @class
   */
  var DepthSubscriber = function DepthSubscriber(depth) {
    if (typeof depth !== 'number' || depth < 0) {
      throw new Error('depth must be 0 or greater');
    }
    var that = this,
      sub = new baja.Subscriber();
    that.$depth = depth;
    that.$subscriber = sub;
    that.$directComps = [];
    sub.attach('added', function (prop) {
      var comp = this.get(prop),
        depth = comp.getType().is('baja:Component') ? that.getDepth(comp) : -1;
      if (depth >= 0) {
        subscribeAll([comp], sub, that.$depth - depth)["catch"](logError);
      }
    });
  };

  /**
   * Returns an array of the currently subscribed components
   * by this 'DepthSubscriber'
   * @returns {Array.<baja.Component>} a copy of the array of
   * components subscribed by this 'DepthSubscriber'
   */
  DepthSubscriber.prototype.getComponents = function () {
    return this.$directComps.slice();
  };

  /**
   * Return true if the 'DepthSubscriber' is empty of subscriptions
   * @returns {Boolean} true if the 'DepthSubscriber' is empty
   */
  DepthSubscriber.prototype.isEmpty = function () {
    return this.$directComps.length === 0;
  };

  /**
   * Get the depth of the given component. If the component given is directly
   * subscribed by this `DepthSubscriber`, the result will be 0, a child
   * component will give 1, etc.
   *
   * If no component parameter given, returns the configured depth (`depth`
   * param to constructor).
   *
   * @param {baja.Complex} [comp]
   * @returns {Number} depth of the given component, or -1 if component is
   * not subscribed by this subscriber. Note that even if the component is
   * a descendant of a component subscribed by this `DepthSubscriber`, if it is
   * outside of the configured subscribe depth, -1 will still be returned.
   */
  DepthSubscriber.prototype.getDepth = function (comp) {
    var directComps = this.$directComps,
      depth = 0,
      $depth = this.$depth;
    if (!comp) {
      return $depth;
    }
    while (comp) {
      if (directComps.indexOf(comp) >= 0) {
        return depth < $depth ? depth : -1;
      }
      depth++;
      comp = comp.getParent();
    }
    return -1;
  };

  /**
   * Attaches events to the backing `Subscriber`.
   */
  DepthSubscriber.prototype.attach = function () {
    var sub = this.$subscriber;
    return sub.attach.apply(sub, arguments);
  };

  /**
   * Detaches events from the backing `Subscriber`.
   */
  DepthSubscriber.prototype.detach = function () {
    var sub = this.$subscriber;
    return sub.detach.apply(sub, arguments);
  };

  /**
   * Return true if the backing `Subscriber` has the given component subscribed.
   *
   * @param {baja.Component} comp
   * @returns {Boolean}
   */
  DepthSubscriber.prototype.isSubscribed = function (comp) {
    return this.$subscriber.isSubscribed(comp);
  };

  /**
   * Subscribes the given components to the depth specified in the constructor.
   * @param {baja.Component|Array.<baja.Component>|Object} comps
   * @returns {Promise} promise to be resolved when the component tree
   * is fully subscribed
   */
  DepthSubscriber.prototype.subscribe = function (comps) {
    comps = baja.objectify(comps, 'comps');
    var that = this,
      arr = toArray(comps.comps);
    return subscribeAll(arr, this.$subscriber, this.$depth).then(function () {
      var directComps = that.$directComps;
      _.each(arr, function (comp) {
        if (directComps.indexOf(comp) < 0) {
          directComps.push(comp);
        }
      });
    }).then(okHandler(comps.ok), failHandler(comps.fail));
  };

  /**
   * Unsubscribes the given components to the depth specified in the
   * constructor.
   * @param {baja.Component|Array.<baja.Component>|Object} comps
   * @returns {Promise} promise to be resolved when the component tree
   * is full unsubscribed
   */
  DepthSubscriber.prototype.unsubscribe = function (comps) {
    comps = baja.objectify(comps, 'comps');
    var that = this,
      arr = toArray(comps.comps);
    return unsubscribeAll(arr, this.$subscriber, this.$depth).then(function () {
      var directComps = that.$directComps;
      _.each(arr, function (comp) {
        var idx = directComps.indexOf(comp);
        if (idx >= 0) {
          directComps.splice(idx, 1);
        }
      });
    }).then(okHandler(comps.ok), failHandler(comps.fail));
  };

  /**
   * Unsubscribes everything currently subscribed, regardless of depth.
   *
   * @returns {Promise}
   */
  DepthSubscriber.prototype.unsubscribeAll = function () {
    return this.$subscriber.unsubscribeAll();
  };
  function okHandler(ok) {
    return function (result) {
      if (ok) {
        ok(result);
      }
      return result;
    };
  }
  function failHandler(fail) {
    return function (err) {
      if (fail) {
        fail(err);
      }
      throw err;
    };
  }
  return DepthSubscriber;
});
