/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/util/ComplexDiff
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/typeUtils'], function (baja, Promise, _, typeUtils) {
  'use strict';

  var isComplex = typeUtils.isComplex,
    isComponent = typeUtils.isComponent,
    isValue = typeUtils.isValue,
    //special constant indicating a slot should be removed rather than changed
    REMOVE_SLOT = {};

  /**
   * A `ComplexDiff` describes a set of changes to be made to a `Complex`
   * instance. These changes include directly setting values of slots,
   * or applying further diffs to nested `Complex`es.
   *
   * @class
   * @alias module:nmodule/webEditors/rc/fe/baja/util/ComplexDiff
   * @param {Object} [map] default set of diffs to apply
   */
  var ComplexDiff = function ComplexDiff(map) {
    this.$map = map || {};
  };

  /**
   * Create a new diff that, when applied, will transform the source Complex
   * to match the target Complex. Any nested Complexes will be transformed as
   * well, although be aware that if the replacement Complex is of a different
   * Type, it will be used to overwrite the target completely with a new
   * instance.
   *
   * Beware when passing in subscribed Components, as this will not perform any
   * sort of depth subscription and you may lose data! See
   * {@link module:nmodule/webEditors/rc/fe/baja/util/DepthSubscriber DepthSubscriber}
   * if you intend to transform subscribed components of unknown depth.
   *
   * @param {baja.Complex} original
   * @param {baja.Complex} transformed
   * @returns {module:nmodule/webEditors/rc/fe/baja/util/ComplexDiff} a diff
   * that will transform source into target
   * @throws {Error} if source and target are not Complexes or are of different
   * types
   */
  ComplexDiff.diff = function (original, transformed) {
    validateSourceAndTarget(original, transformed);
    var diff = new ComplexDiff(),
      sourceSlots = slotNames(original),
      targetSlots = slotNames(transformed),
      toRemove = _.difference(sourceSlots, targetSlots),
      toAdd = _.difference(targetSlots, sourceSlots),
      toChange = _.intersection(sourceSlots, targetSlots);
    toRemove.forEach(function (slot) {
      diff.set(slot, null);
    });
    toAdd.forEach(function (slot) {
      diff.set(slot, transformed.get(slot).newCopy(true));
    });
    toChange.forEach(function (slot) {
      var sourceValue = original.get(slot),
        targetValue = transformed.get(slot);
      if (!sourceValue.equivalent(targetValue)) {
        if (sourceValue.getType().equals(targetValue.getType())) {
          if (isComplex(sourceValue)) {
            diff.set(slot, ComplexDiff.diff(sourceValue, targetValue));
          } else {
            diff.set(slot, targetValue.newCopy(true));
          }
        } else {
          diff.set(slot, targetValue.newCopy(true));
        }
      }
    });
    return diff;
  };

  /**
   * Describe a change to a particular slot. This does not perform any actual
   * changes until `apply` is called.
   *
   * @param {String} slot the slot name to change
   * @param {module:nmodule/webEditors/rc/fe/baja/util/ComplexDiff|baja.Value|null} value
   * the value to add or apply to the slot, or null if the slot should be
   * removed
   */
  ComplexDiff.prototype.set = function (slot, value) {
    if (value === null) {
      this.$map[slot] = REMOVE_SLOT;
    } else if (value instanceof ComplexDiff || isValue(value)) {
      this.$map[slot] = value;
    } else {
      throw new Error('baja.Value or ComplexDiff required');
    }
  };

  /**
   * Takes the changes described by this diff and applies them to the given
   * `Complex`.
   *
   * @param {baja.Complex} complex
   * @param {Object} [params]
   * @param {baja.comm.Batch} [params.batch] the batch to use when applying
   * changes to the `Complex`
   * @returns {Promise.<baja.Complex>} promise to be resolved with the given
   * Complex with the changes applied
   */
  ComplexDiff.prototype.apply = function (complex, params) {
    if (!isComplex(complex)) {
      return Promise.reject(new Error('can only apply a ComplexDiff to a Complex'));
    }
    var reserve = baja.comm.Batch.reserve(params),
      promises = _.map(this.$map, function (value, slot) {
        return applyChangesToTargetComplex(complex, slot, value, reserve);
      });
    return reserve.commit(Promise.all(promises)).then(_.constant(complex));
  };
  function applyChangesToTargetComplex(complex, slot, valueToApply, batch) {
    if (valueToApply instanceof ComplexDiff) {
      return applyDiffToComplex(valueToApply, complex.get(slot), slot, batch);
    } else if (valueToApply === REMOVE_SLOT) {
      if (complex.has(slot)) {
        return removeSlot(complex, slot, batch);
      }
    } else if (complex.has(slot)) {
      return setSlot(complex, slot, valueToApply, batch);
    } else {
      return addSlot(complex, slot, valueToApply, batch);
    }
  }
  function applyDiffToComplex(diff, complex, slot, batch) {
    if (isComplex(complex)) {
      return diff.apply(complex, {
        batch: batch
      });
    } else {
      return die('cannot apply diff "' + slot + '" to a non-Complex slot.');
    }
  }
  function removeSlot(complex, slot, batch) {
    if (complex.getSlot(slot).isFrozen()) {
      return die('cannot remove frozen slot "' + slot + '"');
    }
    return complex.remove({
      slot: slot,
      batch: batch
    });
  }
  function addSlot(complex, slot, value, batch) {
    if (!isComponent(complex)) {
      return die('cannot add a slot to a Struct');
    }
    return complex.add({
      slot: slot,
      value: value,
      batch: batch
    });
  }
  function setSlot(complex, slot, value, batch) {
    return complex.set({
      slot: slot,
      value: value,
      batch: batch
    });
  }

  /**
   * Given a `Complex`, sneak a peek at what the value _would_ be after applying
   * diff to it.
   *
   * If the requested value is accounted for in this diff, this will resolve
   * the value the diff states it should have. Otherwise it will resolve the
   * actual value from the `Complex`.
   *
   * @param {baja.Complex} complex
   * @param {String} slotPath the slot path to look up, minus the `slot:` prefix
   * @returns {Promise} promise to be resolved with the value the `Complex`
   * would have after the diff was applied (can be null if the property is to
   * be removed)
   * @example
   *   var comp = baja.$('baja:Component', {
   *         comp: baja.$('baja:Component', {
   *           simple1: 'nestedSimple1',
   *           simple2: 'nestedSimple2'
   *         })
   *         simple1: 'simple1',
   *         simple2: 'simple2'
   *       }),
   *       diff = new ComplexDiff(),
   *       subDiff = new ComplexDiff();
   *
   *   subDiff.set('simple2', 'newNestedSimple2');
   *   diff.set('simple1', 'newSimple1');
   *   diff.set('comp', subDiff);
   *
   *   diff.getValue(comp, 'simple1'); //resolves 'newSimple1'
   *   diff.getValue(comp, 'simple2'); //resolves 'simple2'
   *   diff.getValue(comp, 'comp/simple1'); //resolves 'nestedSimple1'
   *   diff.getValue(comp, 'comp/simple2'); //resolves 'newNestedSimple2'
   *
   *   diff.apply(comp); //only now are the changes committed
   */
  ComplexDiff.prototype.getValue = function (complex, slotPath) {
    var value = getValue(this, complex, slotPath.split('/'));
    if (value === REMOVE_SLOT) {
      return Promise.resolve(null);
    }
    if (value === null) {
      if (isComponent(complex)) {
        return baja.Ord.make(new baja.SlotPath(slotPath)).get({
          base: complex
        });
      } else {
        return Promise.resolve(walkSlotPath(complex, slotPath));
      }
    }
    return Promise.resolve(value);
  };
  function getValue(diff, complex, names) {
    var name = names[0];
    if (name) {
      var value = diff.$map[name];
      if (value === REMOVE_SLOT) {
        return REMOVE_SLOT;
      }
      if (isValue(value)) {
        return value;
      }
      if (value instanceof ComplexDiff) {
        return getValue(value, complex, names.slice(1));
      }
    }
    return null;
  }

  /**
   * If the value for this slot name is itself another `ComplexDiff`, return it.
   *
   * @param {String} slot
   * @returns {module:nmodule/webEditors/rc/fe/baja/util/ComplexDiff|null} the
   * sub-diff, or `null` if it is not another diff
   */
  ComplexDiff.prototype.getSubDiff = function (slot) {
    var diff = this.$map[slot];
    return diff instanceof ComplexDiff ? diff : null;
  };
  function walkSlotPath(struct, slotPath) {
    var names = slotPath.split('/').reverse(),
      value = struct,
      name;
    while ((name = names.pop()) && value.has(name)) {
      value = value.get(name);
    }
    return value;
  }
  function slotNames(comp) {
    return comp.getSlots().properties().toArray().map(String);
  }
  function die(msg) {
    return Promise.reject(new Error(msg));
  }
  function validateSourceAndTarget(source, target) {
    if (!isComplex(source)) {
      throw new Error('source must be a Complex');
    }
    if (!isComplex(target)) {
      throw new Error('target must be a Complex');
    }
    if (!source.getType().equals(target.getType())) {
      throw new Error('source and target must be the same type');
    }
  }
  return ComplexDiff;
});
