/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/RelTimeEditor
 */
define(['baja!', 'lex!webEditors', 'log!nmodule.webEditors.rc.fe.baja.RelTimeEditor', 'jquery', 'Promise', 'bajaux/events', 'nmodule/webEditors/rc/fe/baja/util/rangeUtils', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'hbs!nmodule/webEditors/rc/fe/baja/template/RelTimeEditor'], function (baja, lexs, log, $, Promise, events, rangeUtils, BaseEditor, tplRelTimeEditor) {
  'use strict';

  var webEditorsLex = lexs[0],
    logError = log.severe.bind(log),
    inRange = rangeUtils.inRange,
    MODIFY_EVENT = events.MODIFY_EVENT;

  /**
   * Structure representing the different fields and their labels. Will get
   * passed to the Handlebars template.
   */
  var FIELDS = [{
    name: 'days',
    label: webEditorsLex.get('RelTimeEditor.label.days', 'd')
  }, {
    name: 'hours',
    label: webEditorsLex.get('RelTimeEditor.label.hours', 'h')
  }, {
    name: 'minutes',
    label: webEditorsLex.get('RelTimeEditor.label.minutes', 'm')
  }, {
    name: 'seconds',
    label: webEditorsLex.get('RelTimeEditor.label.seconds', 's')
  }, {
    name: 'ms',
    label: webEditorsLex.get('RelTimeEditor.label.ms', 'ms')
  }];

  /**
   * Sets the sign of the positive/negative button. Shows/Hides the button and
   * the negative sign label depending on the sign of the value, min facet, and
   * max facet.
   *
   * @private
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/baja/RelTimeEditor} ed
   * @param {Boolean} valPos true for positive, false for negative
   */
  function setSign(ed, valPos) {
    var that = ed,
      dom = that.jq(),
      button = dom.children('button'),
      negLabel = dom.children('.negative-sign'),
      props = that.properties(),
      minPos = (props.getValue('min') || baja.RelTime.make(-1)) >= 0,
      maxPos = (props.getValue('max') || baja.RelTime.make(1)) >= 0;
    button.toggleClass('positive', valPos).toggleClass('negative', !valPos).text(valPos ? '+' : '-');

    // Truth Table
    // val  min  max  showButton  showNeg
    // neg  neg  neg  FALSE       TRUE
    // neg  pos  pos  TRUE        FALSE    invalid: facets contradict loaded val
    // neg  neg  pos  TRUE        FALSE
    // pos  neg  neg  TRUE        FALSE    invalid: facets contradict loaded val
    // pos  pos  pos  FALSE       FALSE
    // pos  neg  pos  TRUE        FALSE

    negLabel.toggle(!valPos && !maxPos);
    button.toggle(!(valPos === minPos && minPos === maxPos));
  }

  /**
   * Loads the fields of the `RelTime` into the `Integer` subeditors.
   *
   * @private
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/baja/RelTimeEditor} ed
   * @param {baja.RelTime} relTime
   */
  function loadFields(ed, relTime) {
    var showDay = ed.$showDay(),
      pos = relTime.getMillis() >= 0 ? 1 : -1,
      values = {
        days: showDay ? relTime.getDaysPart() : 0,
        hours: showDay ? relTime.getHoursPart() : relTime.getDaysPart() * 24 + relTime.getHoursPart(),
        minutes: relTime.getMinutesPart(),
        seconds: relTime.getSecondsPart(),
        ms: relTime.getMillisPart()
      };
    function loadEditorField(ed) {
      var field = ed.jq().data('field'); //days/hours/minutes/seconds/ms
      return ed.load(baja.Integer.make(values[field] * pos));
    }
    return Promise.all(ed.getChildEditors().map(loadEditorField));
  }

  /**
   * A field editor for working with RelTimes.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/RelTimeEditor
   */
  var RelTimeEditor = function RelTimeEditor(params) {
    BaseEditor.call(this, $.extend({
      keyName: 'RelTimeEditor'
    }, params));

    /**
     * A `RelTimeEditor` will fail to validate under the following conditions:
     *
     * - `max` facet is set and the current `RelTime` value is too large
     * - `min` facet is set and the current `RelTime` value is too small
     *
     * Note that facets are converted to numbers before the check, so
     * `Number` values will be treated directly as millis while `RelTime`s will
     * work as well.
     *
     * @method module:nmodule/webEditors/rc/fe/baja/RelTimeEditor#validate
     */
    this.validators().add(function (val) {
      var props = this.properties();
      return inRange(val, {
        min: props.getValue('min'),
        max: props.getValue('max')
      });
    });
  };
  RelTimeEditor.prototype = Object.create(BaseEditor.prototype);
  RelTimeEditor.prototype.constructor = RelTimeEditor;

  /**
   * Return true if day field should be shown (`showDay` facet is true).
   * @private
   * @returns {Boolean}
   */
  RelTimeEditor.prototype.$showDay = function () {
    return this.properties().getValue('showDay') === true;
  };

  /**
   * Return true if seconds field should be shown (`showSeconds` facet is not
   * false).
   * @private
   * @returns {boolean}
   */
  RelTimeEditor.prototype.$showSeconds = function () {
    return this.properties().getValue('showSeconds') !== false;
  };

  /**
   * Return true if milliseconds field should be shown (seconds field is shown,
   * and `showMilliseconds` facet is true).
   * @private
   * @returns {boolean}
   */
  RelTimeEditor.prototype.$showMilliseconds = function () {
    return this.$showSeconds() && this.properties().getValue('showMilliseconds') === true;
  };

  /**
   * Returns true if the editor has its sign button currently toggled to
   * positive.
   *
   * @private
   * @returns {Boolean}
   */
  RelTimeEditor.prototype.$isPositive = function () {
    return this.jq().children('button').hasClass('positive');
  };

  /**
   * Loads `Integer` editors for hours, minutes, seconds, and ms.
   * (Seconds will be hidden if `showSeconds` facet is false, and
   * `showMilliseconds` facet must be true to show milliseconds.)
   *
   * Creates a positive/negative toggle button to change the sign/direction of
   * the time period.
   *
   * Arms change handlers on number inputs to properly increment/decrement
   * other fields when rolling over past 0/60/etc. Arms click handler on button
   * to properly toggle the sign. Both handlers set modified state to true.
   *
   * @param {JQuery} dom
   */
  RelTimeEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.html(tplRelTimeEditor({
      showDay: that.$showDay(),
      showSeconds: that.$showSeconds(),
      showMilliseconds: that.$showMilliseconds()
    }));
    dom.on(MODIFY_EVENT, '.editor', function () {
      that.setModified(true);
      return false;
    });
    dom.on('change', '.editor', function () {
      that.read().then(function (relTime) {
        //TODO: Switchboard ahoy
        if (that.isDestroyed()) {
          return;
        }
        var mePos = that.$isPositive(),
          timePos = relTime.getMillis() >= 0;

        //don't allow for skipping from positive to negative - only clicking
        //the button can do that. just set to 0 instead.
        if (mePos !== timePos) {
          relTime = baja.RelTime.make(0);
        }
        loadFields(that, relTime);
      })["catch"](logError);
      that.setModified(true);
      return false; //don't bubble event for sub-editor - only for myself
    });
    dom.on('click', 'button', function () {
      setSign(that, !that.$isPositive());
      that.setModified(true);
    });
    function buildChildForField(field) {
      var fieldDom = dom.children('.' + field.name);
      return that.buildChildFor({
        value: baja.Integer.DEFAULT,
        dom: fieldDom,
        properties: {
          nativeHtml: true
        },
        formFactor: 'mini'
      }).then(function () {
        //TODO: is there a way to do this with facets instead?
        fieldDom.children('label.postlabel').text(field.label);
      });
    }
    return Promise.all(FIELDS.map(buildChildForField));
  };

  /**
   * Loads the hours, minutes, seconds, and milliseconds fields from the
   * `RelTime` into the different number inputs. Toggles the
   * positive/negative button appropriately.
   *
   * @param {baja.RelTime} value
   */
  RelTimeEditor.prototype.doLoad = function (value) {
    setSign(this, value.getMillis() >= 0);
    return loadFields(this, value);
  };

  /**
   * Reads the hours, minutes, seconds, and milliseconds fields and assembles
   * them into a `baja.RelTime` instance with the sign taken from the
   * positive/negative button's current state.
   *
   * @returns {Promise} promise to be resolved with the constructed
   * `baja.RelTime`, or rejected if any of the fields contain an invalid
   * number value.
   */
  RelTimeEditor.prototype.doRead = function () {
    var that = this,
      pos = that.$isPositive(),
      //build up this object to pass to baja.RelTime.make
      rtObj = {};
    function readEditorField(ed) {
      var field = ed.jq().data('field'); //days/hours/minutes/seconds/ms
      return ed.read().then(function (integer) {
        rtObj[field] = integer.valueOf();
      });
    }
    return Promise.all(that.getChildEditors().map(readEditorField)).then(function () {
      var rt = baja.RelTime.make(rtObj);

      //if negative button is set, flip the sign.
      if (!pos) {
        rt = baja.RelTime.make(rt.getMillis() * -1);
      }
      return rt;
    });
  };

  /**
   * Enables/disables all buttons and number editors.
   *
   * @param {Boolean} enabled
   * @return {Promise} promise to be resolved when all number editors
   * have their enabled state changed
   */
  RelTimeEditor.prototype.doEnabled = function (enabled) {
    this.jq().children('button').prop('disabled', this.isReadonly() || !enabled);
    return this.getChildEditors().setAllEnabled(enabled);
  };

  /**
   * Disables/enables all buttons and number editors.
   *
   * @param {Boolean} readonly
   * @return {Promise} promise to be resolved when all number editors
   * have their readonly state changed
   */
  RelTimeEditor.prototype.doReadonly = function (readonly) {
    this.jq().children('button').prop('disabled', !this.isEnabled() || readonly);
    return this.getChildEditors().setAllReadonly(readonly);
  };

  /**
   * Destroys child editors.
   *
   * @returns {Promise}
   */
  RelTimeEditor.prototype.doDestroy = function () {
    return this.getChildEditors().destroyAll();
  };
  return RelTimeEditor;
});
