/**
 * @copyright 2016 Tridium, Inc. All Rights Reserved.
 * @author Vikram Nagulan
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/PasswordEditor
 */
define(['baja!', 'lex!webEditors', 'Promise', 'bajaux/events', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/servlets/password', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/StringEditor', "bajaScript/baja/obj/Password", 'hbs!nmodule/webEditors/rc/fe/baja/template/PasswordEditor'], function (baja, lexs, Promise, events, compUtils, password, BaseEditor, StringEditor, Password, tplPasswordEditor) {
  'use strict';

  var webEditorsLex = lexs[0],
    MUST_BE_SECURE = webEditorsLex.get('UserPasswordEditor.mustBeSecure'),
    PASSWORD = '',
    SHOULD_VALIDATE = BaseEditor.SHOULD_VALIDATE,
    MODIFY_EVENT = events.MODIFY_EVENT;

  /**
   * Editor for handling `baja:Password` values.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/PasswordEditor
   */
  var PasswordEditor = function PasswordEditor(params) {
    var that = this;
    BaseEditor.apply(this, arguments);

    /**
     * A PasswordEditor will fail to validate under the following conditions:
     *
     * - the editor is operating in an insecure environment - passwords must
     *   not be sent over the wire in cleartext
     * - the two entered passwords do not match
     *
     * @method module:nmodule/webEditors/rc/fe/baja/PasswordEditor#validate
     */
    that.validators().add(function () {
      return that.$isSecure().then(function (secure) {
        if (!secure) {
          throw new Error(MUST_BE_SECURE);
        }
      });
    });
  };
  PasswordEditor.prototype = Object.create(BaseEditor.prototype);
  PasswordEditor.prototype.constructor = PasswordEditor;

  /**
   * Get the password string editor.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/StringEditor}
   */
  PasswordEditor.prototype.$getPasswordEditor = function () {
    return this.jq().children('.password').data('widget');
  };

  /**
   * Check to make sure this password editor is operating in a secure
   * environment. Note that this is a client-side check only to prevent editor
   * from validating. If the station detects passwords from an insecure source
   * it must reject those passwords.
   *
   * @private
   * @returns {Promise}
   */
  PasswordEditor.prototype.$isSecure = function () {
    return password.$isSecure();
  };

  /**
   * @param {JQuery} dom
   * @returns {Promise}
   */
  PasswordEditor.prototype.doInitialize = function (dom) {
    var that = this,
      preLabel = that.properties().get("prelabel"),
      pwdLabel = preLabel ? preLabel.value : PASSWORD;

    //get the input label

    return that.$isSecure().then(function (secure) {
      dom.html(tplPasswordEditor({
        passwordId: that.generateId(),
        password: pwdLabel
      }));
      dom.on(MODIFY_EVENT, '.password', function () {
        that.setModified(true);
        return false;
      });
      dom.attr('title', secure ? '' : MUST_BE_SECURE);
      return new StringEditor({
        properties: {
          inputType: 'password',
          autocomplete: 'off'
        },
        readonly: !secure
      }).initialize(dom.children('.password'));
    });
  };

  /**
   * @param {baja.Simple} pwd the `baja:Password` value to load
   */
  PasswordEditor.prototype.doLoad = function (pwd) {
    var that = this;
    if (!baja.hasType(pwd, 'baja:Password')) {
      return Promise.reject(new Error('baja:Password required'));
    }
    return Promise.resolve(this.isComplexSlotEditor() && this.getComplex()).then(function (complex) {
      if (complex) {
        var parentComp = compUtils.getParentComponent(complex),
          shouldValidate = parentComp && !parentComp.isMounted();
        that.properties().add(SHOULD_VALIDATE, shouldValidate);
      }
      return complex;
    });
  };

  /**
   * Simply read the first password field.
   *
   * @returns {Promise} promise to be resolved with the currently
   * entered password.
   */
  PasswordEditor.prototype.doRead = function () {
    return this.$getPasswordEditor().read().then(function (pw) {
      return Password.DEFAULT.decodeFromString(pw);
    });
  };

  /**
   * Saving a `PasswordEditor` only makes sense when editing a password that
   * belongs to a Complex. Since BajaScript will not encode a `baja:Password`
   * (nor should it), the save operation will be delegated to the
   * server instead.
   *
   * @see module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor
   * @see module:nmodule/webEditors/rc/servlets/password
   * @returns {Promise}
   */
  PasswordEditor.prototype.saveToComplex = function (pw, params) {
    return password.setPassword(pw.encodeToString(), this.getSlot(), this.getComplex());
  };

  /**
   * Destroys "password" editor.
   *
   * @returns {Promise}
   */
  PasswordEditor.prototype.doDestroy = function () {
    return this.$getPasswordEditor().destroy();
  };

  /**
   * Set password editor readonly. Will not make them writable if
   * editor is not working in a secure environment.
   *
   * @param {Boolean} readonly
   * @returns {Promise}
   */
  PasswordEditor.prototype.doReadonly = function (readonly) {
    var that = this;
    return that.$isSecure().then(function (secure) {
      if (!readonly && !secure) {
        return;
      }
      return that.$getPasswordEditor().setReadonly(readonly);
    });
  };

  /**
   * Set password editor enabled. Will not enable them if editor is
   * not working in a secure environment.
   *
   * @param {Boolean} enabled
   * @returns {Promise}
   */
  PasswordEditor.prototype.doEnabled = function (enabled) {
    var that = this;
    return that.$isSecure().then(function (secure) {
      if (enabled && !secure) {
        return;
      }
      return that.$getPasswordEditor().setEnabled(enabled);
    });
  };
  return PasswordEditor;
});
