/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/OrderedMapEditor
 */
define(['baja!', 'lex!', 'underscore', 'bajaux/icon/iconUtils', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/util/htmlUtils'], function (baja, lex, _, iconUtils, BaseEditor, htmlUtils) {
  'use strict';

  var escapeHtml = htmlUtils.escapeHtml;

  /**
   * An editor for working with `OrderedMap`s. Since an `OrderedMap` is not a
   * `baja.Value`, this is not really a field editor; rather more of a UI
   * component for generating select dropdowns on the fly.
   *
   * This editor supports the following Properties:
   *
   * - `lexicon`: have the display values passed through the lexicon of a
   *   particular module.
   * - `sortKeys`: set to `true` to cause the keys to be sorted before adding
   *   them to the dropdown.
   *
   * Note that `baja.OrderedMap` is considered private API.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/OrderedMapEditor
   */
  var OrderedMapEditor = function OrderedMapEditor() {
    BaseEditor.apply(this, arguments);
  };
  OrderedMapEditor.prototype = Object.create(BaseEditor.prototype);
  OrderedMapEditor.prototype.constructor = OrderedMapEditor;

  /**
   * Return the select dropdown element.
   *
   * @private
   * @returns {JQuery}
   */
  OrderedMapEditor.prototype.$getSelect = function () {
    return this.jq().children('select');
  };

  /**
   * Return the icon span.
   *
   * @private
   * @returns {JQuery}
   */
  OrderedMapEditor.prototype.$getIconSpan = function () {
    return this.jq().children(".ux-currentValue-icon");
  };

  /**
   * Creates a select dropdown element.
   *
   * @param {JQuery} dom
   */
  OrderedMapEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.addClass('OrderedMapEditor');

    //TODO: make hbs template
    dom.html('<span class="ux-currentValue-icon"></span><select class="ux-select" />');
    dom.on('change', 'select', function () {
      that.setModified(true);
      that.$updateIcon();
    });
  };

  /**
   * Loads the `baja.OrderedMap` into the select dropdown. Create an `<option>`
   * for each key-value pair in the map.
   *
   * @param {baja.OrderedMap|Object} value the OrderedMap, or an object literal
   * to be converted to an OrderedMap
   * @param {Object} [params] params passed into `load()` method
   * @param {String} [params.selectedKey] a key to preselect
   * @param {*} [params.selectedValue] a value to preselect (overrides
   * `selectedKey` if both are present)
   */
  OrderedMapEditor.prototype.doLoad = function (value, params) {
    if (value.constructor === Object) {
      var map = new baja.OrderedMap();
      Object.keys(value).forEach(function (k) {
        return map.put(k, value[k]);
      });
      value = this.$value = map;
    }
    var that = this,
      props = that.properties(),
      select = that.$getSelect(),
      keys = value.getKeys(),
      p = params || {},
      selectedKey = p.selectedKey,
      selectedValue = p.selectedValue,
      val,
      lexicon = props.getValue('lexicon'),
      lexiconIcons = props.getValue('lexiconIcons');
    if (props.getValue('sortKeys')) {
      keys.sort();
    }
    function buildOption(lex, key) {
      var v = value.get(key),
        icon,
        text = lex ? lex.get(key) : key;
      if (lexiconIcons) {
        icon = lex ? lex.get(key + ".icon") : '';
      }
      if (key === selectedKey) {
        val = key;
      }
      if (selectedValue !== undefined && selectedValue !== null && v === selectedValue) {
        val = key;
      }
      var valueImageString = icon ? ' valueImage="' + icon + '"' : '';
      return '<option class="ux-option"' + valueImageString + ' value="' + escapeHtml(key) + '">' + escapeHtml(text) + '</option>';
    }
    function buildSelect(lex) {
      var options = _.map(keys, function (key) {
        return buildOption(lex, key);
      });
      select.html(options.join(''));
      if (val) {
        select.val(val);
        that.$updateIcon()["catch"](baja.error);
      }
    }
    if (lexicon) {
      return lex.module(lexicon).then(buildSelect);
    } else {
      return buildSelect();
    }
  };

  /**
   *
   * @return {Promise}
   */
  OrderedMapEditor.prototype.$updateIcon = function () {
    var that = this;
    var imageSpan = that.$getIconSpan();
    var image = this.$getSelect().children("option:selected").attr('valueImage');
    if (!image) {
      return Promise.resolve(imageSpan.html(""));
    } else {
      return iconUtils.toHtml(baja.Ord.make(image)).then(function (html) {
        imageSpan.html(html);
      });
    }
  };

  /**
   * Returns the currently selected key.
   * @returns {String} currently selected key, or undefined if none
   */
  OrderedMapEditor.prototype.getSelectedKey = function () {
    return this.$getSelect().val();
  };

  /**
   * Sets the currently selected key. This will trigger a `MODIFY_EVENT`.
   *
   * @param {String} key
   * @param {Boolean} [silent] set to true if the `MODIFY_EVENT` should not be
   * triggered.
   */
  OrderedMapEditor.prototype.setSelectedKey = function (key, silent) {
    var select = this.$getSelect();
    select.val(key);
    if (!silent) {
      select.trigger('change');
    }
  };

  /**
   * Returns the currently selected value.
   *
   * @returns {*}
   */
  OrderedMapEditor.prototype.getSelectedValue = function () {
    var value = this.value();
    return value && value.get(this.getSelectedKey());
  };

  /**
   * Returns an array of all keys that map to the given value.
   *
   * @param {*} value
   * @returns {Array.<String>} keys
   */
  OrderedMapEditor.prototype.getKeysFor = function (value) {
    var map = this.value(),
      keys = map.getKeys();
    return _.compact(_.map(keys, function (key) {
      return map.get(key) === value && key;
    }));
  };

  /**
   * Sets the currently selected value. This will trigger a `MODIFY_EVENT`.
   *
   * @param {*} value
   * @param {Boolean} [silent] set to true if the `MODIFY_EVENT` should not be
   * triggered.
   */
  OrderedMapEditor.prototype.setSelectedValue = function (value, silent) {
    var that = this,
      map = that.value();
    _.some(map.getKeys(), function (key) {
      if (map.get(key) === value) {
        var select = that.$getSelect();
        select.val(key);
        if (!silent) {
          select.trigger('change');
        }
      }
    });
  };

  /**
   * Get the currently selected value from the `baja.OrderedMap`. This will not
   * just return the `value` attribute of the currently selected `<option>`
   * but will retrieve the actual value from the map.
   *
   * @returns {undefined|*} the currently selected value, or undefined if
   * no values are available from the currently loaded `baja.OrderedMap`
   * @throws {Error} if somehow the selected key does not exist in the loaded
   * map
   */
  OrderedMapEditor.prototype.doRead = function () {
    var map = this.value(),
      key = this.getSelectedKey();
    if (!map || map.getSize() === 0) {
      return undefined;
    } else if (map.contains(key)) {
      return map.get(key);
    } else {
      throw new Error('selected key ' + key + ' not contained in map');
    }
  };

  /**
   * Enables/disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  OrderedMapEditor.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables/enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  OrderedMapEditor.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };

  /**
   * @returns {Promise}
   */
  OrderedMapEditor.prototype.doDestroy = function () {
    this.jq().removeClass('OrderedMapEditor');
    return BaseEditor.prototype.doDestroy.apply(this, arguments);
  };
  return OrderedMapEditor;
});
