function _slicedToArray(r, e) { return _arrayWithHoles(r) || _iterableToArrayLimit(r, e) || _unsupportedIterableToArray(r, e) || _nonIterableRest(); }
function _nonIterableRest() { throw new TypeError("Invalid attempt to destructure non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
function _iterableToArrayLimit(r, l) { var t = null == r ? null : "undefined" != typeof Symbol && r[Symbol.iterator] || r["@@iterator"]; if (null != t) { var e, n, i, u, a = [], f = !0, o = !1; try { if (i = (t = t.call(r)).next, 0 === l) { if (Object(t) !== t) return; f = !1; } else for (; !(f = (e = i.call(t)).done) && (a.push(e.value), a.length !== l); f = !0); } catch (r) { o = !0, n = r; } finally { try { if (!f && null != t["return"] && (u = t["return"](), Object(u) !== u)) return; } finally { if (o) throw n; } } return a; } }
function _arrayWithHoles(r) { if (Array.isArray(r)) return r; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/NumericEditor
 */
define(['baja!', 'baja!baja:UnitConversion', 'jquery', 'Promise', 'nmodule/webEditors/rc/fe/baja/util/numberUtils', 'nmodule/webEditors/rc/fe/baja/util/rangeUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/fe/baja/util/FloatingPointHelper', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/util/htmlUtils', 'hbs!nmodule/webEditors/rc/fe/baja/template/NumericEditor'], function (baja, types, $, Promise, numberUtils, rangeUtils, typeUtils, FloatingPointHelper, BaseEditor, htmlUtils, tplNumericEditor) {
  'use strict';

  var convertUnitFrom = numberUtils.convertUnitFrom,
    getDisplayUnits = numberUtils.getDisplayUnits,
    getRangeInfo = rangeUtils.getRangeInfo,
    inRange = rangeUtils.inRange,
    isNumber = typeUtils.isNumber,
    selectOnFocus = htmlUtils.selectOnFocus,
    selectAll = htmlUtils.selectAll;
  var DEFAULT_PROPERTIES = {
    forceSign: false,
    precision: 2,
    showSeparators: false
  };

  /**
   * A field editor for working with double/float values.
   *
   * This editor supports the following Widget Properties:
   *
   * - `forceSign`: set to `true` to always show a `+` when number is positive
   * - `max`: set a maximum valid value. If `nativeHtml` is set, will add a
   *   `max` property to the `input` tag.
   * - `min`: set a minimum valid value. If `nativeHtml` is set, will add a
   *   `min` property to the `input` tag.
   * - `nativeHtml`: set to `true` to use `input type="number"` rather than
   *   a `text` input.
   * - `precision`: specify the number of decimal places to show
   * - `prelabel`: specify some text to show just before the text input
   * - `postlabel`: specify some text to show just after the text input
   * - `showSeparators`: set to `true` to show separators
   * - `step`: if `nativeHtml` is set, will add a `step` property to the `input`
   *   tag.
   * - `unitConversion`: if given, perform the specified English/metric
   *   conversion when displaying and reading the value. If `units` is not
   *   given, this will have no effect. This can be a `baja:UnitConversion`
   *   instance, a number corresponding to one of its ordinals, or a string
   *   corresponding to one of its tags.
   * - `units`: show the given unit symbol next to the text input.
   * - `zeroPad`: specify the minimum number of digits to show before the
   *   decimal place
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/NumericEditor
   */
  var NumericEditor = function NumericEditor(params) {
    var that = this;
    BaseEditor.call(that, $.extend({
      keyName: 'NumericEditor'
    }, params));

    /**
     * A NumericEditor will fail to validate under the following conditions:
     *
     * - `max` facet is set and the current number value is too large
     * - `min` facet is set and the current number value is too small
     * @method module:nmodule/webEditors/rc/fe/baja/NumericEditor#validate
     */
    that.validators().add(function (val) {
      return that.$inRange(val);
    });
  };
  NumericEditor.prototype = Object.create(BaseEditor.prototype);
  NumericEditor.prototype.constructor = NumericEditor;

  /**
   * Returns a helper object to assist in parsing/format floating point numbers.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/util/FloatingPointHelper}
   */
  NumericEditor.prototype.$getHelper = function () {
    var that = this,
      helper = that.$helper;
    if (!helper) {
      var value = that.value(),
        ctor = value !== null && value.constructor;
      helper = new FloatingPointHelper(ctor || baja.Double, $.extend({}, DEFAULT_PROPERTIES, that.properties().toValueMap()));
      that.$helper = helper;
    }
    return helper;
  };

  /**
   * Return the text input element.
   *
   * @private
   * @returns {jQuery}
   */
  NumericEditor.prototype.$getTextElement = function () {
    return this.jq().children('input');
  };

  /**
   * Get the text to go into the label immediately before the text input. This
   * will be overridden by a `prelabel` Property, if present.
   *
   * @private
   * @returns {String}
   */
  NumericEditor.prototype.$getPreLabelText = function () {
    return '';
  };

  /**
   * Get the text to go into the label immediately after the text input. This
   * will be overridden by a `postlabel` Property, if present.
   *
   * @private
   * @returns {Promise.<String>}
   */
  NumericEditor.prototype.$getPostLabelText = function () {
    var that = this,
      postlabel = [];
    return Promise.all([that.$getDisplayUnits(), getRangeInfo(getProps(that))]).then(function (_ref) {
      var _ref2 = _slicedToArray(_ref, 2),
        units = _ref2[0],
        rangeInfo = _ref2[1];
      if (units !== null && !units.isPrefix()) {
        postlabel.push(units.getSymbol());
      }
      if (rangeInfo.rangeStr) {
        postlabel.push(rangeInfo.rangeStr);
      }
      return postlabel.join(' ');
    });
  };

  /**
   * @private
   * @returns {baja.Unit|null} the value of the `units` property
   */
  NumericEditor.prototype.$getUnits = function () {
    return this.properties().getValue('units');
  };

  /**
   * @private
   * @returns {baja.FrozenEnum|Number} the value of the `unitConversion` property
   */
  NumericEditor.prototype.$getUnitConversion = function () {
    return this.properties().getValue('unitConversion');
  };

  /**
   * Get the units configured on this editor. If a `unitConversion` property is
   * also present, the units will be converted as specified.
   *
   * @private
   * @returns {Promise} promise to be resolved with `baja.Units`, or `null` if
   * no units present
   */
  NumericEditor.prototype.$getDisplayUnits = function () {
    return getDisplayUnits(this.$getUnits(), this.$getUnitConversion());
  };

  /**
   *  @private
   *  @param {baja.Simple} val
   *  @returns {Promise} to be rejected if out of min/max range
   */
  NumericEditor.prototype.$inRange = function (val) {
    return inRange(val, getProps(this));
  };
  NumericEditor.prototype.$selectAll = function () {
    selectAll(this.$getTextElement());
  };
  NumericEditor.prototype.requestFocus = function () {
    BaseEditor.prototype.requestFocus.apply(this, arguments);
    this.$selectAll();
  };

  /**
   * Creates a number input field.
   *
   * @param {JQuery} dom
   */
  NumericEditor.prototype.doInitialize = function (dom) {
    //TODO: revisit use of fieldWidth facet. Why limit text length instead of setting max?
    //TODO: scientific notation?
    var that = this;
    return that.$getPostLabelText().then(function (postLabelText) {
      var props = that.properties(),
        prelabel = props.getValue('prelabel') || that.$getPreLabelText(),
        postlabel = props.getValue('postlabel') || postLabelText;
      dom.on('input', 'input', function () {
        that.setModified(true);
      });
      selectOnFocus(that, 'input', function () {
        return that.$selectAll();
      });
      return that.$getDisplayUnits().then(function (units) {
        if (units) {
          if (units.isPrefix()) {
            prelabel = prelabel || units.getSymbol();
          } else {
            postlabel = postlabel || units.getSymbol();
          }
        }
        dom.html(tplNumericEditor({
          prelabel: prelabel,
          postlabel: postlabel
        }));
        var input = dom.children('input');
        if (props.getValue('nativeHtml')) {
          input.prop('type', 'number');
          if (props.has('max')) {
            input.prop('max', props.getValue('max'));
          }
          if (props.has('min')) {
            input.prop('min', props.getValue('min'));
          }
          if (props.has('step')) {
            input.prop('step', props.getValue('step'));
          }
        } else {
          input.prop('type', 'text');
        }
      });
    });
  };

  /**
   * Loads the given number into the input field.
   *
   * @param {Number} value
   */
  NumericEditor.prototype.doLoad = function (value) {
    var that = this;
    return that.$getHelper().format(value).then(function (str) {
      that.$getTextElement().val(str);
    });
  };

  /**
   * Retrieves the currently entered number from the input field. Resultant
   * number will be either `baja.Float` or `baja.Double` depending on the
   * value currently loaded into the editor.
   *
   * @returns {Promise.<baja.Double|baja.Float>} the currently entered number
   */
  NumericEditor.prototype.doRead = function () {
    var that = this,
      helper = that.$getHelper(),
      result = helper.parse(that.$getTextElement().val().toLowerCase());
    if (isNaN(result)) {
      return Promise.resolve(result);
    }

    //convert from user's preferred unit system to real number
    return convertUnitFrom(result.valueOf(), that.$getUnits(), that.$getUnitConversion()).then(function (converted) {
      return helper.toType(converted);
    });
  };

  /**
   * Enables or disables the input field.
   *
   * @param {Boolean} enabled
   */
  NumericEditor.prototype.doEnabled = function (enabled) {
    this.$getTextElement().prop('disabled', !enabled);
  };

  /**
   * Sets the input field to readonly/not readonly.
   *
   * @param {Boolean} readonly
   */
  NumericEditor.prototype.doReadonly = function (readonly) {
    this.$getTextElement().prop('readonly', readonly);
  };

  /**
   * Ensure that min/max are of double type for range calculations.
   * @param {module:nmodule/webEditors/rc/fe/baja/NumericEditor} ed
   * @returns {Object}
   */
  function getProps(ed) {
    var props = ed.properties().toValueMap(),
      min = props.min,
      max = props.max;
    if (isNumber(min)) {
      props.min = +min;
    }
    if (isNumber(max)) {
      props.max = +max;
    }
    return props;
  }
  return NumericEditor;
});
