/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/IntegerEditor
 */
define(['baja!', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/util/IntegralHelper', 'nmodule/webEditors/rc/fe/baja/util/rangeUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/fe/baja/NumericEditor'], function (baja, Promise, _, IntegralHelper, rangeUtils, typeUtils, NumericEditor) {
  'use strict';

  var inRange = rangeUtils.inRange,
    isNumber = typeUtils.isNumber,
    DEFAULT_RADIX = 10;

  /**
   * A field editor for working with integer/long values.
   *
   * This editor supports the following Properties:
   *
   * - `forceSign`: set to `true` to always show a `+` when number is positive
   * - `radix`: set the radix for parsing the number (default 10)
   * - `showSeparators`: set to `true` to show separators (will delegate to
   *   `Number#toLocaleString()`
   * - `zeroPad`: specify the minimum number of digits to show before the
   *   decimal place
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/NumericEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/IntegerEditor
   */
  var IntegerEditor = function IntegerEditor(params) {
    NumericEditor.call(this, _.extend({
      keyName: 'IntegerEditor'
    }, params));
  };
  IntegerEditor.prototype = Object.create(NumericEditor.prototype);
  IntegerEditor.prototype.constructor = IntegerEditor;

  /**
   * Returns a helper object to assist in parsing/format integral numbers.
   *
   * @private
   * @returns {module:nmodule/webEditors/rc/fe/baja/util/IntegralHelper}
   */
  IntegerEditor.prototype.$getHelper = function () {
    var that = this,
      helper = that.$helper;
    if (!helper) {
      var value = that.value(),
        ctor = value !== null && value.constructor;
      helper = new IntegralHelper(ctor || baja.Long, getProps(that));
      that.$helper = helper;
    }
    return helper;
  };

  /**
   * @private
   * @returns {number} the configured radix property, if present (defaulting to 10).
   */
  IntegerEditor.prototype.$getRadix = function () {
    var radix = this.properties().getValue('radix');
    if (!isNumber(radix) || radix < 2 || radix > 36) {
      return DEFAULT_RADIX;
    }
    return +radix;
  };

  /**
   * @private
   * @returns {Promise.<string>} postlabel text including relevant units, radix,
   * and range information.
   */
  IntegerEditor.prototype.$getPostLabelText = function () {
    var props = getProps(this),
      min = props.min,
      max = props.max,
      units = props.units,
      text = [],
      radix = this.$getRadix();
    if (baja.hasType(units, 'baja:Unit') && !units.isPrefix()) {
      text.push(units.getSymbol());
    }
    if (radix !== DEFAULT_RADIX) {
      text.push('Radix=' + radix);
    }
    if (isValidLowerBound(min) || isValidUpperBound(max)) {
      text.push('[' + numberToString(isNumber(min) ? min : baja.Long.MIN_VALUE, radix) + ' - ' + numberToString(isNumber(max) ? max : baja.Long.MAX_VALUE, radix) + ']');
    }
    return Promise.resolve(text.join(' '));
  };
  IntegerEditor.prototype.$inRange = function (val) {
    var props = getProps(this),
      min = props.min,
      max = props.max;
    if (isValidLowerBound(min)) {
      props.min = baja.Long.make(min);
    }
    if (isValidUpperBound(max)) {
      props.max = baja.Long.make(max);
    }
    return inRange(val, props);
  };

  /**
   * Override to always return 0 (no unit conversion allowed on integers).
   * @returns {number} 0
   */
  IntegerEditor.prototype.$getUnitConversion = function () {
    return 0;
  };

  /**
   * If `radix` facet is set, forces `input` type to be `text` (`number` does
   * not support hex characters) and displays radix after field.
   *
   * @param {JQuery} dom
   */
  IntegerEditor.prototype.doInitialize = function (dom) {
    var that = this;
    return NumericEditor.prototype.doInitialize.apply(that, arguments).then(function () {
      if (that.$getRadix() !== DEFAULT_RADIX) {
        dom.children('input').prop('type', 'text');
      }
    });
  };

  /**
   * @param {baja.Simple} num
   * @returns {boolean} true if valid range lower bound (a number not min/-inf)
   */
  function isValidLowerBound(num) {
    if (!isNumber(num)) {
      return false;
    }
    var str = num.encodeToString();
    return str !== 'min' && str !== '-inf';
  }

  /**
   * @param {baja.Simple} num
   * @returns {boolean} true if valid range upper bound (a number not max/+inf)
   */
  function isValidUpperBound(num) {
    if (!isNumber(num)) {
      return false;
    }
    var str = num.encodeToString();
    return str !== 'max' && str !== '+inf';
  }
  function getProps(ed) {
    return _.chain(ed.properties().toValueMap()).omit('unitConversion').extend({
      precision: 0
    }).value();
  }

  /**
   * @param {baja.Simple} num
   * @param {number} radix
   * @returns {String}
   */
  function numberToString(num, radix) {
    var str = num.encodeToString();

    // handle Float/Double displaying -inf/+inf
    if (str === '-inf' || str === 'min') {
      return 'min';
    } else if (str === '+inf' || str === 'max') {
      return 'max';
    }
    return baja.Long.make(num).valueOf().toString(radix);
  }
  return IntegerEditor;
});
