function _toConsumableArray(r) { return _arrayWithoutHoles(r) || _iterableToArray(r) || _unsupportedIterableToArray(r) || _nonIterableSpread(); }
function _nonIterableSpread() { throw new TypeError("Invalid attempt to spread non-iterable instance.\nIn order to be iterable, non-array objects must have a [Symbol.iterator]() method."); }
function _unsupportedIterableToArray(r, a) { if (r) { if ("string" == typeof r) return _arrayLikeToArray(r, a); var t = {}.toString.call(r).slice(8, -1); return "Object" === t && r.constructor && (t = r.constructor.name), "Map" === t || "Set" === t ? Array.from(r) : "Arguments" === t || /^(?:Ui|I)nt(?:8|16|32)(?:Clamped)?Array$/.test(t) ? _arrayLikeToArray(r, a) : void 0; } }
function _iterableToArray(r) { if ("undefined" != typeof Symbol && null != r[Symbol.iterator] || null != r["@@iterator"]) return Array.from(r); }
function _arrayWithoutHoles(r) { if (Array.isArray(r)) return _arrayLikeToArray(r); }
function _arrayLikeToArray(r, a) { (null == a || a > r.length) && (a = r.length); for (var e = 0, n = Array(a); e < a; e++) n[e] = r[e]; return n; }
/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/*jshint browser: true */ /* eslint-env browser */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/IconEditor
 */
define(['baja!', 'bajaux/icon/iconUtils', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/slotUtils', 'css!nmodule/icons/sprite/sprite'], function (baja, iconUtils, Promise, _, BaseEditor, slotUtils) {
  'use strict';

  var getSlotIcon = slotUtils.getSlotIcon,
    toElements = iconUtils.toElements,
    IMAGE_ERROR_BEHAVIOR_PROPERTY = "imageErrorBehavior",
    DRAGGABLE_PROPERTY = "draggable";

  ////////////////////////////////////////////////////////////////
  // Support functions
  ////////////////////////////////////////////////////////////////

  /**
   * Given an array of URI strings, convert them to a `baja.Icon` (converting
   * `/module/` to `module://` ORD syntax).
   *
   * @inner
   * @param {Array.<String|baja.Ord>} uris
   * @returns {baja.Icon}
   */
  function urisToIcon(uris) {
    return baja.Icon.make(_.map(uris, function (uri) {
      if (typeof uri === 'string') {
        return uri.replace('/module/', 'module://');
      } else {
        return uri;
      }
    }));
  }

  /**
   * Get/make the appropriate `baja.Icon` instance for the loaded value.
   *
   * @inner
   * @param {baja.Icon|baja.Value|baja.Slot|Array.<String>} value
   * @returns {baja.Icon}
   */
  function toIcon(value) {
    if (baja.hasType(value, 'baja:Icon')) {
      return value;
    }
    if (baja.hasType(value)) {
      return value.getType().getIcon();
    }
    if (value instanceof baja.Slot) {
      return urisToIcon(getSlotIcon(value));
    }
    if (_.isArray(value)) {
      return urisToIcon(value);
    }
  }

  /**
   * Gets the dimensions of the icon based upon the underlying style.
   * @param {JQuery} target
   * @returns {Array.<number>}
   */
  function getNaturalIconDimensions(target) {
    return [target.width(), target.height()];
  }

  ////////////////////////////////////////////////////////////////
  // IconEditor
  ////////////////////////////////////////////////////////////////

  /**
   * A display-only editor for working with `Icon`s and Images.
   *
   * It supports the following `bajaux` Properties:
   *
   * - `sync`: if `true`, will force the image to be fully downloaded before
   *   `load()` completes. Use this if you are relying on the icon image
   *   dimensions for layout purposes.
   * - `imageErrorBehavior`: this determines what happens when there is an image
   *   load error. If set to 'hide', the image will be hidden (default
   *   behavior). If set to 'placeholder', the image will have a fallback load
   *   in the place based upon the styling of the imageError class applied to
   *   the internal span of this element.
   * - `draggable`: If set to `true` or `false` and an `img` is used for display purposes, the `img`
   *   attribute for `draggable` will be set to the provided value.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/IconEditor
   */
  var IconEditor = function IconEditor() {
    BaseEditor.apply(this, arguments);
  };
  IconEditor.prototype = Object.create(BaseEditor.prototype);
  IconEditor.prototype.constructor = IconEditor;

  /**
   * Add `IconEditor` class.
   *
   * @param {JQuery} dom
   */
  IconEditor.prototype.doInitialize = function (dom) {
    dom.addClass('IconEditor');
    var wrapper = document.createElement('div');
    wrapper.className = 'wrapper';
    dom.html(wrapper);
  };

  /**
   * Load in an icon for display. This function can take a value of several
   * different kinds:
   *
   * - a `baja.Icon` directly
   * - any other `baja.Value` - its Type's icon will be shown
   * - a `baja.Slot`: a Property will show that property's Type, while an
   *   Action or Topic will show a generic `action.png` or `topic.png` icon
   * - an array of image URIs, which will be converted directly
   *
   * @param {baja.Icon|baja.Value|baja.Slot|Array.<String>} icon
   */
  IconEditor.prototype.doLoad = function (icon) {
    var that = this,
      sync = this.properties().getValue('sync'),
      jq = this.jq().children('.wrapper'),
      imageLoadedPromiseResolve,
      imageLoadedPromiseReject,
      draggableProp = that.properties().getValue(DRAGGABLE_PROPERTY);
    if (!jq.length) {
      return;
    }

    // eslint-disable-next-line
    this.$imageLoadedPromise = new Promise(function (resolve, reject) {
      imageLoadedPromiseResolve = resolve;
      imageLoadedPromiseReject = function imageLoadedPromiseReject(e) {
        var img = jq.find("img"),
          imageErrorBehaviorProp = that.properties().getValue(IMAGE_ERROR_BEHAVIOR_PROPERTY);
        if (imageErrorBehaviorProp === "placeholder") {
          img.closest("span").toggleClass("imageError", true);
        } else {
          img.css('visibility', 'hidden');
        }
        reject(e);
      };
    });
    return toElements(toIcon(icon), sync).then(function (elements) {
      var _jq$empty$;
      (_jq$empty$ = jq.empty()[0]).append.apply(_jq$empty$, _toConsumableArray(elements));
      var img = jq.find("img"),
        src = img.attr("src"),
        isSpritesheetIcon = src && src.match("data:image/png;base64");
      if (isSpritesheetIcon) {
        imageLoadedPromiseResolve();
      } else {
        img.on("load", imageLoadedPromiseResolve);
        img.on("error", imageLoadedPromiseReject);

        //only set the draggableProp when the property is provided
        if (draggableProp === false || draggableProp === true) {
          img.attr(DRAGGABLE_PROPERTY, draggableProp);
        }
      }
    });
  };

  /**
   * Gets the dimensions once the Icon is loaded.
   * Resolves on success with width and height of Icon.
   * Resolves with the dimensions of the placeholder image if image load failed
   * and imageErrorBehavior is placeholder.
   * Resolves to null if image load failed and imageErrorBehavior is hide.
   * @returns {Promise.<Array.<number>|null>}
   */
  IconEditor.prototype.dimensions = function () {
    var that = this;
    return this.$imageLoadedPromise.then(function () {
      var jq = that.jq(),
        img = jq.find("img"),
        isSpritesheetIcon = img.attr("src").match("data:image/png;base64");
      if (isSpritesheetIcon) {
        return getNaturalIconDimensions(img.closest("span"));
      } else {
        return that.$getNaturalImageDimensions(img);
      }
    })["catch"](function () {
      var jq = that.jq(),
        img = jq.find("img"),
        imageErrorBehaviorProp = that.properties().getValue(IMAGE_ERROR_BEHAVIOR_PROPERTY);
      if (imageErrorBehaviorProp === "placeholder") {
        return getNaturalIconDimensions(img.closest("span"));
      } else {
        return null;
      }
    });
  };

  /**
   * Gets the natural height / width of the image. The image should
   * be loaded prior to this being called. This will not get correct dimensions
   * for an Icon.
   * @param {JQuery} img the picture element
   * @returns {Array.<number>}
   */
  IconEditor.prototype.$getNaturalImageDimensions = function (img) {
    return [Number(img[0].naturalWidth), Number(img[0].naturalHeight)];
  };

  /**
   * Resolve the currently loaded icon. Note that if any other value than a
   * `baja.Icon` was loaded in `doLoad`, it will be converted to a `baja.Icon`
   * when read.
   *
   * @returns {baja.Icon}
   */
  IconEditor.prototype.doRead = function () {
    return toIcon(this.value());
  };

  /**
   * Remove `IconEditor` class.
   */
  IconEditor.prototype.doDestroy = function () {
    this.jq().removeClass('IconEditor');
  };
  return IconEditor;
});
