/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/FrozenEnumEditor
 */
define(['baja!', 'jquery', 'Promise', 'underscore', 'nmodule/webEditors/rc/fe/baja/BaseEditor', 'nmodule/webEditors/rc/fe/baja/util/rangeUtils', 'nmodule/webEditors/rc/util/htmlUtils', 'Promise'], function (baja, $, Promise, _, BaseEditor, rangeUtils, htmlUtils) {
  'use strict';

  var escapeHtml = htmlUtils.escapeHtml;

  /**
   * A field editor for working with `FrozenEnum`s.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/FrozenEnumEditor
   */
  var FrozenEnumEditor = function FrozenEnumEditor(params) {
    BaseEditor.call(this, $.extend({
      keyName: 'FrozenEnumEditor'
    }, params));
  };
  FrozenEnumEditor.prototype = Object.create(BaseEditor.prototype);
  FrozenEnumEditor.prototype.constructor = FrozenEnumEditor;

  /**
   * Get the select element.
   *
   * @private
   * @returns {JQuery}
   */
  FrozenEnumEditor.prototype.$getSelect = function () {
    return this.jq().children('select');
  };

  /**
   * Creates a select dropdown.
   *
   * @param {JQuery} dom
   */
  FrozenEnumEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.append('<select class="ux-select"/>');
    dom.on('change', 'select', function () {
      that.setModified(true);
      that.$removeInvalidOption();
    });
  };

  /**
   * When the loaded enum is set to an invalid option, that option is
   * added to the select in `doLoad()`.  When the user selects
   * another (valid) option, `$removeInvalidOption()` removes the
   * additional invalid option that was originally added during
   * `doLoad()`.
   */
  FrozenEnumEditor.prototype.$removeInvalidOption = function () {
    var validOptions = this.$getValidOptions();
    if (!validOptions) {
      return;
    }
    var select = this.$getSelect(),
      kids = select.children();
    _.each(kids, function (kid) {
      // Don't remove it if it's selected
      if (select.val() !== kid.value) {
        // Do remove the option if it's not a valid option.
        if (!validOptions.getOrdinals().contains(parseInt(kid.value))) {
          // Remove this child and then exit because only one
          // option should be invalid.
          kid.remove();
        }
      }
    });
  };

  /**
   * Returns the `uxValidOptions` facet set on the editor, only if it is present
   * and is of type `baja:String` and can be decoded by <code>baja:EnumSet</code>.
   *
   * @private
   * @returns {baja.EnumRange} range facet set, or undefined if not present
   * or not `EnumRange`
   */
  FrozenEnumEditor.prototype.$getValidOptions = function () {
    var range = this.properties().getValue('uxValidOptions');
    if (baja.hasType(range, 'baja:String')) {
      return baja.EnumSet.DEFAULT.decodeFromString(range);
    }
  };

  /**
   * Returns the <option/> HTML string for the specified ordinal and display
   * value.
   *
   * @private
   * @param {String|Number} ordinal - enum ordinal used as the value for the
   *    option element.
   * @param {String} display - display value of the option
   * @returns {String}
   */
  FrozenEnumEditor.prototype.$toOption = function (ordinal, display) {
    return '<option class="ux-option" value="' + ordinal + '">' + escapeHtml(display) + '</option>';
  };

  /**
   * Assembles the `Enum` into a set of options to load into the select
   * dropdown.
   *
   * @param {baja.FrozenEnum} value
   * @returns {Promise}
   */
  FrozenEnumEditor.prototype.doLoad = function (value) {
    var that = this,
      // Determine the correct options to be used for the options elements.
      validOptions = this.$getValidOptions(),
      range = validOptions || value.getRange(),
      found = false;
    // Iterate through the correct ordinals and construct the options elements.
    return Promise.all(_.map(range.getOrdinals(), function (o) {
      // While iterating, verify that the currently selected ordinal is
      // included in the list of options.
      if (value.getOrdinal() === o) {
        found = true;
      }
      if (_.isFunction(value.get(o).getDisplayTag)) {
        var tag = value.get(o).getDisplayTag();
        return that.$toOption(o, tag);
      } else {
        return rangeUtils.getEnumRangeDisplay(o, value.getRange()).then(function (displayTag) {
          return that.$toOption(o, displayTag);
        });
      }
    })).then(function (results) {
      var html = results.join('');

      // If the uxValidOptions did not include value.getOrdinal() then add another
      // option to include the erroneous ordinal (otherwise the user will be
      // presented with an unselected / incorrect display).
      if (validOptions && !found) {
        html = html + that.$toOption(value.getOrdinal(), value.getDisplayTag());
      }
      that.$getSelect().html(html).val(value.getOrdinal());
    });
  };

  /**
   * Returns an instance of the `Enum` with ordinal selected according to
   * the selected option in the dropdown.
   *
   * @returns {baja.Enum} the currently selected `Enum` instance
   */
  FrozenEnumEditor.prototype.doRead = function () {
    var ordinal = parseInt(this.$getSelect().val(), 10);
    return this.value().get(ordinal);
  };

  /**
   * Enables or disables the select dropdown.
   *
   * @param {Boolean} enabled
   */
  FrozenEnumEditor.prototype.doEnabled = function (enabled) {
    this.$getSelect().prop('disabled', this.isReadonly() || !enabled);
  };

  /**
   * Disables or enables the select dropdown.
   *
   * @param {Boolean} readonly
   */
  FrozenEnumEditor.prototype.doReadonly = function (readonly) {
    this.$getSelect().prop('disabled', !this.isEnabled() || readonly);
  };
  return FrozenEnumEditor;
});
