/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/DateEditor
 */
define(['baja!', 'baja!baja:Month', 'nmodule/webEditors/rc/fe/baja/BaseEditor'], function (baja, types, BaseEditor) {
  'use strict';

  /**
   * A field editor for working with Dates.
   *
   * @class
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/DateEditor
   */
  var DateEditor = function DateEditor() {
    BaseEditor.apply(this, arguments);
  };
  DateEditor.prototype = Object.create(BaseEditor.prototype);
  DateEditor.prototype.constructor = DateEditor;
  /**
   * Returns the date input element.
   *
   * @private
   * @returns {jQuery}
   */
  DateEditor.prototype.$getDateElement = function () {
    return this.jq().children('input');
  };

  /**
   * Creates a date input element. (In browsers that do not support `date`
   * type input fields, this will result in a `text` input.
   *
   * @param {JQuery} dom
   */
  DateEditor.prototype.doInitialize = function (dom) {
    var that = this;
    dom.append('<input type="date">');
    dom.on('change', 'input', function () {
      that.setModified(true);
    });
  };

  /**
   * Loads a `baja.Date` into the date input element. If the browser only
   * supports `text` inputs, this will be a string in ISO 8601 `yyyy-MM-dd`
   * format.
   *
   * @param {baja.Date} value
   * @returns {Promise} promise to be resolved if the value is loaded,
   * or rejected if the `baja.Date` could not be encoded to string
   */
  DateEditor.prototype.doLoad = function (value) {
    var that = this;

    //TODO: i18n
    return value.toDateString({
      textPattern: 'YYYY-MM-DD'
    }).then(function (str) {
      that.$getDateElement().val(str);
    });
  };

  /**
   * Reads the date value from the date input element and constructs a
   * `baja.Date` from it.
   *
   * @returns {baja.Date} the read `baja.Date` value
   * @throws {Error} if the input contains an invalid date value.
   */
  DateEditor.prototype.doRead = function () {
    var val = this.$getDateElement().val();
    try {
      return baja.Date.DEFAULT.decodeFromString(val);
    } catch (e) {
      throw new Error('invalid date format: "' + val + '"');
    }
  };

  /**
   * Enables or disables the date input element.
   *
   * @param {Boolean} enabled
   */
  DateEditor.prototype.doEnabled = function (enabled) {
    this.$getDateElement().prop('disabled', !enabled || this.isReadonly());
  };

  /**
   * Enables or disables the date input element.
   *
   * @param {Boolean} readonly
   */
  DateEditor.prototype.doReadonly = function (readonly) {
    this.$getDateElement().prop('disabled', readonly || !this.isEnabled());
  };
  return DateEditor;
});
