/**
 * @copyright 2015 Tridium, Inc. All Rights Reserved.
 * @author Logan Byam
 */

/**
 * API Status: **Private**
 * @module nmodule/webEditors/rc/fe/baja/ComplexSlotEditor
 */
define(['baja!', 'log!nmodule.webEditors.rc.fe.baja.ComplexSlotEditor', 'Promise', 'underscore', 'bajaux/mixin/batchSaveMixin', 'nmodule/js/rc/tinyevents/tinyevents', 'nmodule/webEditors/rc/fe/baja/util/compUtils', 'nmodule/webEditors/rc/fe/baja/util/facetsUtils', 'nmodule/webEditors/rc/fe/baja/util/slotUtils', 'nmodule/webEditors/rc/fe/baja/util/typeUtils', 'nmodule/webEditors/rc/wb/mixin/mixinUtils'], function defineComplexSlotEditor(baja, log, Promise, _, batchSaveMixin, tinyevents, compUtils, facetsUtils, slotUtils, typeUtils, mixinUtils) {
  'use strict';

  var canWriteSlot = compUtils.canWriteSlot,
    closest = compUtils.closest,
    getMergedSlotFacets = compUtils.getMergedSlotFacets,
    slotPathFromAncestor = compUtils.slotPathFromAncestor;
  var applyFacets = facetsUtils.applyFacets;
  var hasMixin = mixinUtils.hasMixin;
  var toCssClass = slotUtils.toCssClass;
  var isComplex = typeUtils.isComplex,
    isComponent = typeUtils.isComponent,
    isSimple = typeUtils.isSimple;
  var logSevere = log.severe.bind(log);

  //TODO: do we want to automatically put component into subscription? or require that it be subscribed first?
  //TODO: can we get change events from a non-mounted component?
  /**
   * Subscribes for changes to the given slot on the component. If the complex
   * is not a mounted component, this will simply resolve without doing
   * anything.
   *
   * @private
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor} ed
   * @param {baja.Complex} complex
   * @param {baja.Slot} slot
   * @returns {Promise} promise to be resolved once subscription is
   * complete
   */
  function doSubscribe(ed, complex, slot) {
    var comp = closest(complex, 'baja:Component');
    if (!isComponent(comp) || !comp.isMounted()) {
      return Promise.resolve();
    }
    var sub = ed.$$subscriber = new baja.Subscriber(),
      slotPath = slotPathFromAncestor(comp, complex),
      //c.f. BComplex#getPropertyInParentComponent
      componentSlot = comp.getSlot(slotPath[0] || slot);
    var oldSlotName = componentSlot && String(componentSlot);
    function isReady() {
      return ed.isInitialized() && !ed.isModified() && componentSlot;
    }
    sub.attach('changed', function (prop) {
      if (isReady() && prop === componentSlot) {
        ed.load(complex.get(slot)).then(function () {
          return Promise.all(ed.emit('changed', slot));
        })["catch"](logSevere);
      }
    });
    sub.attach('removed', function (prop) {
      if (isReady() && prop === componentSlot) {
        ed.emit('removed', prop);
      }
    });
    sub.attach('renamed', function (prop, oldName) {
      var propName = prop.getName();
      if (isReady() && oldName === oldSlotName) {
        ed.emit('renamed', prop, oldName);
        oldSlotName = propName;
        ed.jq().removeClass(toCssClass(oldName)).addClass(toCssClass(propName)).data('slot', propName);
      }
    });
    sub.attach('flagsChanged', function (prop) {
      if (prop === componentSlot) {
        ed.setReadonly(!canWriteSlot(comp, prop));
      }
    });
    return sub.subscribe(comp);
  }

  /**
   * Validates the complex and slot used for this editor. Applies slot facets
   * to the editor. Subscribes for changes if the complex is a mounted
   * component.
   *
   * @private
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor} ed
   * @param {baja.Complex} complex
   * @param {baja.Slot|String} slot
   * @returns {Promise} promise to be resolved after the complex is loaded
   * and subscribed, or rejected if a valid complex and slot not given or
   * the subscribe fails
   */
  function loadComplex(ed, complex, slot) {
    return Promise["try"](function () {
      if (!isComplex(complex)) {
        throw new Error('complex property required');
      } else if (!slot) {
        throw new Error('slot property required');
      } else if (!complex.has(slot)) {
        throw new Error('unknown slot "' + slot + '"');
      } else {
        slot = complex.getSlot(slot);
        ed.$complex = complex;
        ed.$slot = slot;
        return ed.$getSlotFacets ? ed.$getSlotFacets(complex, slot) : getMergedSlotFacets(complex, slot);
      }
    }).then(function (facets) {
      applyFacets(facets, ed);
      return doSubscribe(ed, complex, slot);
    });
  }

  /**
   * Applies the additional complex editing behavior to the editor instance.
   *
   * @private
   * @inner
   * @param {module:nmodule/webEditors/rc/fe/baja/BaseEditor} ed
   */
  function applyMixin(ed) {
    var _destroy = ed.destroy,
      _initialize = ed.initialize,
      _doSave = ed.doSave;
    ed.$nativeBatchSaveSupport = hasMixin(ed, 'batchSave');
    batchSaveMixin(ed);

    /**
     * After the usual `initialize()` behavior, a `ComplexSlotEditor` will
     * set itself to readonly if the slot has its `readonly` flag set to
     * true.
     *
     * @name module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor#initialize
     * @function
     * @see module:nmodule/webEditors/rc/fe/baja/BaseEditor#initialize
     */
    ed.initialize = function (dom) {
      var _this = this;
      var complex = this.getComplex();
      var slot = this.getSlot();
      return _initialize.apply(this, arguments).then(function () {
        if (!canWriteSlot(complex, slot)) {
          _this.setReadonly(true);
        }
        dom.addClass(toCssClass(slot)).data('slot', slot);
        return null; //squelch "promise not returned" from setReadonly above
      });
    };

    //TODO: new batchSave mixin goes here
    //TODO: need edit by ref semantics?
    /**
     * Sets the slot value on the `Complex` to the value read from the
     * sub-editor - essentially committing the changes to the station (if
     * the `Complex` is mounted).
     *
     * If `BaseEditor#saveToComplex()` is implemented on the wrapped editor,
     * it will be given a chance to perform the work instead.
     *
     * @name module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor#doSave
     * @function
     * @param {baja.Value} readValue the value read from the sub-editor
     * @param {Object} [params] any params passed to the `save` method
     * @param {baja.comm.Batch} [params.batch] a batch to use when setting the
     * slot value
     * @returns {Promise} to be resolved if the slot value on the
     * `Complex` could be set successfully
     * @see module:nmodule/webEditors/rc/fe/baja/BaseEditor#doSave
     * @see module:nmodule/webEditors/rc/fe/baja/BaseEditor#saveToComplex
     */
    ed.doSave = function (readValue, params) {
      var that = this,
        complex = that.getComplex(),
        slot = that.getSlot(),
        saved = typeof that.saveToComplex === 'function' && that.saveToComplex(readValue, params),
        progressCallback = params && params.progressCallback;
      if (saved) {
        return saved;
      }
      var saveProm = null,
        commitProm;

      // eslint-disable-next-line promise/param-names,promise/avoid-new
      commitProm = new Promise(function (readyToCommit) {
        if (that.$nativeBatchSaveSupport) {
          //if our wrapped editor had batchSave support, we have to give it a
          //chance to do its native batching before we notify our external
          //save() caller - e.g., MultiSheet - that we're actually done with
          //the batch after calling complex.set() in the next step.
          saveProm = _doSave.call(that, readValue, _.extend({}, params, {
            progressCallback: function progressCallback(msg) {
              if (msg === batchSaveMixin.COMMIT_READY) {
                readyToCommit();
              }
            }
          }));
        } else {
          saveProm = _doSave.call(that, readValue, params);
          readyToCommit();
        }
      }).then(function () {
        //we're almost ready to signal the caller that it's safe to commit.
        var prom = isSimple(readValue) && complex.set({
          slot: slot,
          value: readValue,
          batch: params && params.batch
        });
        if (progressCallback) {
          progressCallback('commitReady');
        }
        return prom;
      });
      return Promise.all([saveProm, commitProm]);
    };

    /**
     * If this editor is working on a mounted component, will stop any
     * subscriptions that this editor added to that component.
     *
     * @name module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor#destroy
     * @function
     * @returns {Promise} promise to be resolved after sub-editor is
     * destroyed, and any active subscriptions have been stopped.
     * @see module:nmodule/webEditors/rc/fe/baja/BaseEditor#destroy
     */
    ed.destroy = function () {
      var sub = this.$$subscriber,
        comp = closest(this.getComplex(), 'baja:Component');
      return _destroy.apply(this, arguments).then(function () {
        return sub && isComponent(comp) && sub.unsubscribe(comp)["catch"](logSevere);
      });
    };

    /**
     * Returns true, because this editor will be used to write changes back
     * to a Complex instance.
     *
     * @memberOf module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor#
     * @returns {boolean}
     */
    ed.isComplexSlotEditor = function () {
      return true;
    };

    /**
     * Get the Complex instance to which changes will be saved.
     *
     * @memberOf module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor#
     * @returns {baja.Complex}
     */
    ed.getComplex = function () {
      return this.$complex;
    };

    /**
     * Get the Slot on the Complex to which changes will be saved.
     *
     * @memberOf module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor#
     * @returns {baja.Slot}
     */
    ed.getSlot = function () {
      return this.$complex.getSlot(this.$slot);
    };
  }
  var making = false;

  /**
   * A editor for editing a slot on a `Complex`. Note that this editor
   * should not be instantiated directly - use the `make()` method instead.
   *
   * @extends module:nmodule/webEditors/rc/fe/baja/BaseEditor
   * @alias module:nmodule/webEditors/rc/fe/baja/ComplexSlotEditor
   * @mixin
   * @mixes tinyevents
   *
   * @param {Function} Super the super constructor we're actually initializing,
   * e.g. `StringEditor`
   * @param {Object} [params] params object to be passed to super constructor
   */
  function makeComplexSlotEditor(Super, params) {
    if (!making) {
      throw new Error('cannot instantiate directly ' + '(use ComplexSlotEditor.make() instead)');
    }
    var ed = new Super(params);
    tinyevents(ed);
    applyMixin(ed);
    return ed;
  }

  /**
   * Creates a new `ComplexSlotEditor` instance. It will extend, and behave
   * like, the given `Editor` constructor, but includes additional functionality
   * specifically for editing slots on a `Complex`.
   *
   * If the `Complex` is a mounted component, it will be subscribed at this
   * point for updates. Any time the given slot on the `Complex` changes, its
   * value will be read and loaded into the editor (as long as the
   * editor does not have user-entered changes).
   *
   * @param {Function} Super `Editor` class to instantiate.
   * @param {Object} params
   * @param {baja.Complex} params.complex the `Complex` instance on which to
   * edit a slot
   * @param {baja.Slot|String} params.slot the slot to edit
   * @returns {Promise} promise to be resolved after the editor has
   * been instantiated and the `Complex` has been subscribed for updates (if
   * applicable).
   */
  makeComplexSlotEditor.make = function (Super, params) {
    params = params || {};
    var ed;
    making = true;
    try {
      ed = makeComplexSlotEditor(Super, params);
    } finally {
      making = false;
    }
    return loadComplex(ed, params.complex, params.slot).then(_.constant(ed));
  };
  return makeComplexSlotEditor;
});
