/*
  This widget demonstrates how to create powerful visualizations using open web technology.
  Please note, this isn't a complete example nor is this part of Tridium's commercial 
  offering to be used onsite. It's merely a code example to show what can be done.

  This gauge uses a really powerful open source visualization library called 'd3'...

  http://d3js.org/

  There are a plethora of great web technologies out there. We look forward to seeing how
  you incorporate them into Niagara 4 :¬)))).
*/
define(['baja!',
        'baja!control:ControlPoint,baja:StatusNumeric',
        'underscore', // Underscore is a great JavaScript utility library: http://underscorejs.org/
        'bajaux/Widget',
        'bajaux/mixin/subscriberMixIn',
        'd3',
        'css!nmodule/docDeveloper/rc/bajaux/examples/LinearGaugeStyle'], function (
        baja,  
        types,
        _,
        Widget,
        subscriberMixIn,
        d3) {

  'use strict';

  var options = {
    left: 20,
    top: 10
  };

  var LinearGauge = function () {
    var that = this;

    Widget.apply(that, arguments);
    subscriberMixIn(that);

    that.properties().addAll([
    {
      name: "barColor",
      value: "steelblue",
      typeSpec: "gx:Color"
    },
    {
      name: "autoScale",
      value: true
    },
    {
      name: "autoScaleStep",
      value: 10
    },
    {
      name: "max",
      value: 100
    },
    {
      name: "min",
      value: 0
    } 
    ]);
  };

  LinearGauge.prototype = Object.create(Widget.prototype);
  LinearGauge.prototype.constructor = LinearGauge;

  function makeModel(widget) {
    var value = widget.value(),
        props = widget.properties(),
        min = widget.$min = widget.$min === undefined ? props.getValue("min") : widget.$min,
        max = widget.$max = widget.$max === undefined ? props.getValue("max") : widget.$max,
        autoScale = props.getValue("autoScale"),
        autoScaleStep = props.getValue("autoScaleStep");
    
    if (value === undefined || value === null) {
      value = 0;
    }
    // If this is a control point then get its out value...
    else if (value.getType().is("control:ControlPoint")) {
      value = value.getOut().getValue();
    }
    // If this is a StatusNumeric then get its value...
    else if (value.getType().is("baja:StatusNumeric")) {
      value = value.getValue();
    }
    else {
      // Just hope we have a value...
      value = parseFloat(value);
    }

    if (!_.isNumber(value)) {
      value = 0;
    }

    if (autoScale) {
      // Figure out the minimum scale.
      while (value < min) {
        min -= autoScaleStep;
      }

      // Figure out the maximum scale.
      while (value > max) {
        max += autoScaleStep;
      }
    }

    return {
      max: max,
      min: min,
      data: [{
        value: value
      }]
    };  
  }

  function render(widget) {
    var jq = widget.jq(),
        width = (jq.width() || 800) - options.left * 2,
        height = 50,
        model = makeModel(widget),
        x, 
        y, 
        gauge, 
        bk, 
        grid, 
        axis,
        bar;

    x = d3.scale.linear()
        .domain([model.min, model.max])
        .range([0, width]);

    y = d3.scale.linear()
        .range([0, height]);

    gauge = d3.select(widget.jq()[0])
        .select(".example-linear-gauge")
        .select("g");
    
    // Render the background
    bk = gauge.selectAll(".example-linear-grid-background")
      .data(model.data);    

    bk.enter()
      .append("rect")
      .attr("class", "example-linear-grid-background");

    bk.attr("width", width)
      .attr("height", height);

    // The background has a grid on top.
    grid = gauge.selectAll(".example-linear-grid")
      .data(model.data);   

    grid.enter()
      .append("g")
      .attr("class", "example-linear-grid");

    grid.attr("transform", "translate(0," + height + ")")
      .call(d3.svg.axis().scale(x).ticks(50).tickSize(-height))
        .selectAll(".example-linear-tick")
          .data(x.ticks(10), function(d) { return d; })
        .exit()
          .classed("example-linear-minor", true);

    // The background has an axis drawn at the bottom of it.
    axis = gauge.selectAll(".example-linear-axis")
      .data(model.data);       

    axis.enter()
      .append("g")
      .attr("class", "example-linear-axis");

    axis.attr("transform", "translate(0," + height + ")")
      .call(d3.svg.axis().scale(x).ticks(10));

    // The data bar is drawn on top.
    bar = gauge.selectAll(".example-linear-bar")
      .data(model.data);  

    bar.enter()
      .append("g")
      .append("rect")
      .attr("class", "example-linear-bar")
      .attr("x", 0)
      .attr("y", 0);

    bar.attr("height", height / 2)
      .transition()
      .attr("fill", widget.properties().getValue("barColor"))
      .attr("width", function (d) { 
        var w = x(d.value); 
        if (w > width) {
          w = width;
        }
        else if (w < 0) {
          w = 0;
        }
        return w;
      });    
  }

  LinearGauge.prototype.doInitialize = function (jq) {
    var that = this;

    jq.addClass("example-linear-gauge-outer");

    d3.select(jq[0])
      .append('svg')
      .attr('top', 0)
      .attr('left', 0)
      .attr('width', "100%")
      .attr('height', "100%")
      .attr('class', 'example-linear-gauge')
      .append('g')
      .attr('transform', 'translate(' + options.left + ',' + options.top + ')');

    that.getSubscriber().attach("changed", function() {
      render(that);
    });  

    render(that);
  };

  LinearGauge.prototype.doLoad = function (value) {
    render(this);
  };

  LinearGauge.prototype.doLayout =
    LinearGauge.prototype.doChanged = function() {
      render(this);
    };

  LinearGauge.prototype.doDestroy = function () {
    this.jq().removeClass("example-linear-gauge-outer");
  };

  return LinearGauge;
});