/*
  Dashboards in Niagara 4 enable users to modify a Px page without having to use
  a full graphical editor. The data for a Dashboard is stored on the Station for
  the user and the position of the Widget on the Px page.

  In order to use this feature, a user must add a DashboardPane (from the Dashboard
  palette) to a Px page via the Px Editor. A Web Widget is then added to the content of 
  the Dashboard Pane.

  To make a widget dashboardable...

  - Add a widget property with the 'dashboard' attribute set to true.
  - When a dashboard is saved, all properties marked with a truthy dashboard
    attribute are saved.
  - When a dashboard is loaded, all properties are restored.
  - Remember to use the 'readonly' and 'hidden' attributes if you don't want a user
    to see dashboard properties in an editor.
  - If you need to know whether your widget is running in a dashboard, a special 'dashboard'
    property is added to the widget when it's loaded. If this property is true, the widget is
    running in a dashboard.
  - When a widget is loaded on a Dashboard, two commands are dynamically added onto the
    dashboard...
    - Clear Dashboard Data: wipes the widget of any dashboard data. This effectively resets
      the widget back to its default state without any dashboard data.
    - Save: the standard bajaux save command. If the widget is modified, the save command
      is enabled.

  This widget also demonstrates how drag and drop works! Try dragging and dropping anything
  onto the widget.          
*/
define(['bajaux/Widget',
        'bajaux/dragdrop/dragDropUtils',
        'css!nmodule/docDeveloper/rc/bajaux/examples/DashboardWidgetStyle'], function (
        Widget,
        dragDropUtils) {

  'use strict';

  var exampleDbClass = "example-dashboard-widget",
      highlightClass = "example-dashboard-widget-drag-over",
      dbOrdPropName = "dashboardOrd";

  var DashboardWidget = function () {
    var that = this;
    Widget.apply(that, arguments);

    // This is a special property that can be added to see if a widget is running on a dashboard
    // or not. A lot of the time this property shouldn't be needed as a developer should just work
    // with the properties that have been loaded into the widget.
    that.properties().add({
      name: "dashboard",
      value: false,
      hidden: true,
      readonly: true,
      transient: true
    });

    // A property marked with the `dashboard` attribute. When the dashboard is saved, this
    // property information will be saved away to the Station for the user.
    that.properties().add({
      name: dbOrdPropName,
      value: "",
      dashboard: true,
      readonly: true
    });
  };

  DashboardWidget.prototype = Object.create(Widget.prototype);
  DashboardWidget.prototype.constructor = DashboardWidget;

  DashboardWidget.prototype.$append = function (str) {
    // Append some new text to the widget's 'pre' DOM element...
    var jq = this.jq().children("pre");
    jq.text(jq.text() + str + "\n");
    return this;
  };

  function updateFromDrop(widget, dataTransfer) {
    // Handle the drag...
    return dragDropUtils.fromClipboard(dataTransfer)
      .then(function (envelope) {
        switch (envelope.getMimeType()) {
          case 'niagara/navnodes':
            envelope.toJson()
              .then(function (json) {
                // Although multiple items could have been dragged and dropped on, 
                // we're just going to use the first item in the list.
                var obj = json && json[0];

                if (obj && obj.ord) {
                  // Setting this property will automatically mark the widget as modified
                  // due to the fact it's in a Dashboard Pane.
                  widget.properties().setValue(dbOrdPropName, obj.ord);

                  widget.$append("")
                        .$append("New Dashboard ORD...")
                        .$append(obj.ord);          
                }
              });
        }
      });
  }

  DashboardWidget.prototype.doInitialize = function (jq) {
    var that = this;
    jq.html("<pre></pre>");

    jq.addClass(exampleDbClass);

    that.$append("Initialized Dashboard Widget...")
        .$append("Is the Widget on a Dashboard Pane: " + that.properties().getValue("dashboard"))
        .$append("")
        .$append("Dashboard ORD: " + that.properties().getValue(dbOrdPropName))
        .$append("Please drag and drop something from the\nNav Tree onto this Widget...");

    // Set up drag and drop.
    jq.on("dragover", function(e) {
        jq.addClass(highlightClass);
        e.preventDefault();
      })
      .on("dragleave", function(e) {
        jq.removeClass(highlightClass);
        e.preventDefault();
      })
      .on("drop", function(e) {
        jq.removeClass(highlightClass);

        // Do the drop...
        updateFromDrop(that, e.originalEvent.dataTransfer);
        
        e.preventDefault();
        e.stopPropagation();
      });    
  };

  DashboardWidget.prototype.doDestroy = function () {
    var jq = this.jq();
    jq.removeClass(exampleDbClass);
    jq.removeClass(highlightClass);
  };

  return DashboardWidget;
});
