/**
 * @copyright 2015, Tridium, Inc. All Rights Reserved.
 * @author Gareth Johnson
 */

/*jshint browser: true*/

/**
 * API Status: **Private**
 * @module nmodule/docDeveloper/rc/JsPlaygroundWidget
 */
define([
  'baja!',
  'lex!docDeveloper',
  'Promise',
  'jquery',
  'bajaux/Widget',
  'bajaux/commands/Command',
  'bajaux/mixin/subscriberMixIn',
  'nmodule/webEditors/rc/wb/mixin/textEditorMixIn',
  'dialogs',
  'hbs!nmodule/docDeveloper/rc/JsPlaygroundSavedTemplate',
  'css!nmodule/docDeveloper/rc/JsPlaygroundStyle'], function (
  baja,
  lexicons,
  Promise,
  $,
  Widget,
  Command,
  subscriberMixIn,
  textEditorMixIn,
  dialogs,
  savedTemplate) {

  'use strict';

  var lex = lexicons[0],
      idPropName = "id";

  /**
   * A playground for JavaScript code! This is used for learning about using Niagara's
   * HTML5 features.
   *
   * @class
   * @alias module:nmodule/docDeveloper/rc/JsPlaygroundWidget
   * @extends module:bajaux/Widget
   */
  var JsPlaygroundWidget = function () {
    var that = this;
    Widget.apply(that, arguments);

    that.properties().add({
      name: idPropName,
      value: "",
      hidden: true,
      readonly: true,
      transient: true,
      parameter: true // Can be loaded from a view parameter.
    });

    subscriberMixIn(that);
    textEditorMixIn(that);
  };

  JsPlaygroundWidget.prototype = Object.create(Widget.prototype);
  JsPlaygroundWidget.prototype.constructor = JsPlaygroundWidget;

  JsPlaygroundWidget.prototype.doLoad = function (playground) {
    var that = this,
        help = playground.getHelp().toString();

    that.$initEditor("javascript");
    that.$setEditorReadOnly(false);

    // If there's help then show the command.
    if (help && help !== baja.Ord.DEFAULT.toString()) {
      that.getCommandGroup().add(new Command({
        module: "docDeveloper",
        lex: "commands.help",
        func: function () {
          window.niagara.env.hyperlink(help);
        }
      }));
    }

    return playground.load("" + that.properties().getValue(idPropName))
      .then(function (text) {
        that.$setEditorText(text);

        if (playground.getEnableSave()) {
          that.$addSaveCmd();

          that.$onEditorChange(function () {
            that.setModified(true);
          });
        } 

        that.$addRunCmd();
      });
  };

  JsPlaygroundWidget.prototype.doSave = function () {
    var that = this,
        playground = that.value();

    if (playground.getEnableSave()) {
      return playground.save(that.$getEditorText())
        .then(function (id) {
          var ordStr = that.$toOrdFromId(id),
              uriStr = that.$toUriFromId(id);

          return dialogs.showOk({
            title: lex.get("saved"), 
            content: savedTemplate({
              savedMsg: lex.get("saved.message"),
              uriTitle: lex.get("saved.uri"),
              ordTitle: lex.get("saved.ord"),
              uri: uriStr,
              ord: ordStr
            }),
            buttons: [{
              name: "goToLink",
              displayName: lex.get("saved.goToLink"),
              handler: function () {
                window.niagara.env.hyperlink(ordStr);
              }
            }]         
          }).promise();
        });
    }
  };

  /**
   * Return an ORD string from the specified Id.
   *
   * @private
   * 
   * @param  {String} id The id for the saved JavaScript.
   * @return {String} A ORD String with the embedded id as a parameter.
   */
  JsPlaygroundWidget.prototype.$toOrdFromId = function (id) {
    return this.value().getNavOrd().relativizeToSession().toString() + 
      "|view:docDeveloper:JsPlaygroundWidget?id=" +
      encodeURIComponent(id);
  };

  /**
   * Return a URI from the specified Id.
   *
   * @private
   * 
   * @param  {String} id The id for the saved JavaScript.
   * @return {String} A URI with the embedded id as a parameter.
   */
  JsPlaygroundWidget.prototype.$toUriFromId = function (id, origin) {
    return (origin || window.location.origin) + 
      "/ord/" + 
      encodeURI(this.value().getNavOrd().relativizeToSession().toString()) + 
      "%7Cview:docDeveloper:JsPlaygroundWidget?id=" +
      encodeURIComponent(id);
  };

  return JsPlaygroundWidget;
});
