/**
 * @copyright 2020 Tridium, Inc. All Rights Reserved.
 * @author JJ Frankovich
 */

/* jshint browser: true */ /* eslint-env browser */

define(['baja!', 'underscore'], function (baja, _) {
  'use strict';

  var each = _.each;

  /**
   * API Status: **Private**
   * @exports nmodule/bajaui/rc/util/pxUtils
   */
  var exports = {};
  exports.$blinkingElements = [];
  exports.$blinkInterval = null;
  exports.$blinkCounter = 0;
  exports.$elementVisibility = 'hidden';
  var blinkFunction = function blinkFunction() {
    // visible for 600 ms, hidden for 200 ms
    exports.$blinkCounter = ++exports.$blinkCounter % 8;
    exports.$elementVisibility = exports.$blinkCounter >= 2 ? 'inherit' : 'hidden';
    for (var i = 0; i < exports.$blinkingElements.length; i++) {
      if (document.body.contains(exports.$blinkingElements[i])) {
        exports.$blinkingElements[i].style.visibility = exports.$elementVisibility;
      } else {
        // remove element from array if no longer found in DOM
        exports.$blinkingElements.splice(i, 1);
        if (exports.$blinkingElements.length === 0) {
          clearInterval(exports.$blinkInterval);
          exports.$blinkInterval = null;
          exports.$blinkCounter = 0;
          exports.$elementVisibility = 'hidden';
          return;
        }
        i--;
      }
    }
  };

  /**
   * @param {module:nmodule/bajaui/rc/model/UxModel} model
   * @param {object} targetMap a mapping of Px Property names to their widget
   * paths (arrays of slot names)
   * @param {object} valueMap a mapping of Px Property names to their overridden
   * values
   */
  exports.applyPxProperties = function (model, targetMap, valueMap) {
    if (!targetMap || !valueMap) {
      return;
    }
    each(targetMap, function (_ref, name) {
      var targets = _ref.targets;
      each(targets, function (path) {
        var prop = path.pop();
        var kid = model.get(path);
        if (kid) {
          var value = valueMap[name];
          if (typeof value !== 'undefined') {
            kid.getProperties()[prop] = valueMap[name];
          }
        }
      });
    });
  };

  /**
   * Make DOM elements start or stop blinking. By maintaining a central list of the blinking items,
   * this will prevent the blinking items from blinking at different times.
   *
   * @see {module:nmodule/hx/com/tridium/hx/px/hxPx.toggleBlink}
   */
  exports.updateBlink = function (elements, blink) {
    if (!blink && exports.$blinkingElements.length === 0) {
      return;
    }
    for (var i = 0; i < elements.length; i++) {
      if (!blink && elements[i]) {
        var index = exports.$blinkingElements.indexOf(elements[i]);
        if (index > -1) {
          exports.$blinkingElements.splice(index, 1);
          if (exports.$blinkingElements.length === 0) {
            clearInterval(exports.$blinkInterval);
            exports.$blinkInterval = null;
            exports.$blinkCounter = 0;
            exports.$elementVisibility = 'hidden';
          }
          elements[i].style.visibility = 'inherit';
        }
      } else {
        if (!elements[i] || exports.$blinkingElements.indexOf(elements[i]) > -1) {
          continue;
        }

        // sync newly blinking element up with the rest of the blinking elements
        elements[i].style.visibility = exports.$elementVisibility;
        exports.$blinkingElements.push(elements[i]);
        if (!exports.$blinkInterval && exports.$blinkingElements.length === 1) {
          exports.$blinkInterval = setInterval(blinkFunction, 100);
        }
      }
    }
  };

  /**
   * @param {string|baja.Ord} ord
   * @returns {baja.Ord} an ORD with the view query removed, if present
   */
  exports.removeViewQuery = function (ord) {
    if (!ord) {
      return baja.Ord.DEFAULT;
    }
    var queries = new baja.OrdQueryList();
    baja.Ord.make(ord).parse().getCursor().each(function (value) {
      var scheme = value.getSchemeName();
      if (scheme !== 'view') {
        queries.add(value);
      }
    });
    return baja.Ord.make(queries);
  };

  /**
   * Recursively update all bajaux Properties, and Binding Properties, that are
   * type ORD, with the appropriate ORD variable substitutions.
   * @param {module:nmodule/bajaui/rc/model/UxModel} model
   * @param {baja.Facets|object} [variables]
   * @returns {Promise.<module:nmodule/bajaui/rc/model/UxModel>} to be resolved
   * with the updated model
   */
  exports.substituteOrds = function (model, variables) {
    var props = model.getProperties();
    Object.keys(props).forEach(function (name) {
      var value = props[name];
      if (isOrd(value)) {
        props[name] = value.substitute(variables);
      }
    });
    return Promise.all(model.getBindingList().getBindings().map(function (binding) {
      return substituteOrds(binding, variables);
    })).then(function () {
      return Promise.all(model.getKids().map(function (kid) {
        return exports.substituteOrds(kid, variables);
      }));
    }).then(function () {
      return model;
    });
  };
  function substituteOrds(comp, variables) {
    var props = comp.getSlots().properties().toArray();
    return Promise.all(props.map(function (slot) {
      var val = comp.get(slot);
      return isOrd(val) && comp.set({
        slot: slot,
        value: val.substitute(variables)
      });
    }));
  }
  function isOrd(v) {
    return baja.hasType(v, 'baja:Ord');
  }
  return exports;
});
